<?php
/**
 * Handle fuel logs
 */
function handleFuelLogs($conn, $method) {
    $id = $_GET['id'] ?? null;
    
    try {
        switch($method) {
            case 'GET':
                if ($id) {
                    $stmt = $conn->prepare("
                        SELECT fl.*, v.vehicle_number, v.vehicle_type
                        FROM fuel_logs fl
                        JOIN transport_vehicles v ON fl.vehicle_id = v.vehicle_id
                        WHERE fl.fuel_log_id = ?
                    ");
                    $stmt->execute([$id]);
                    echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                } else {
                    $vehicleId = $_GET['vehicle_id'] ?? null;
                    $startDate = $_GET['start_date'] ?? null;
                    $endDate = $_GET['end_date'] ?? null;
                    
                    $where = [];
                    $params = [];
                    
                    if ($vehicleId) {
                        $where[] = "fl.vehicle_id = ?";
                        $params[] = $vehicleId;
                    }
                    
                    if ($startDate) {
                        $where[] = "fl.log_date >= ?";
                        $params[] = $startDate;
                    }
                    
                    if ($endDate) {
                        $where[] = "fl.log_date <= ?";
                        $params[] = $endDate;
                    }
                    
                    $whereClause = $where ? 'WHERE ' . implode(' AND ', $where) : '';
                    
                    $stmt = $conn->prepare("
                        SELECT fl.*, v.vehicle_number, v.vehicle_type
                        FROM fuel_logs fl
                        JOIN transport_vehicles v ON fl.vehicle_id = v.vehicle_id
                        $whereClause
                        ORDER BY fl.log_date DESC, fl.created_at DESC
                    ");
                    $stmt->execute($params);
                    echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
                }
                break;

            case 'POST':
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (empty($data['vehicle_id']) || empty($data['fuel_quantity']) || empty($data['fuel_cost'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Vehicle ID, fuel quantity, and cost are required']);
                    return;
                }
                
                // Get previous odometer reading to calculate distance
                $prevStmt = $conn->prepare("
                    SELECT odometer_reading 
                    FROM fuel_logs 
                    WHERE vehicle_id = ? 
                    ORDER BY log_date DESC, created_at DESC 
                    LIMIT 1
                ");
                $prevStmt->execute([$data['vehicle_id']]);
                $prevLog = $prevStmt->fetch(PDO::FETCH_ASSOC);
                
                $currentOdometer = $data['odometer_reading'] ?? 0;
                $distanceCovered = 0;
                $fuelEfficiency = 0;
                
                if ($prevLog && $currentOdometer > $prevLog['odometer_reading']) {
                    $distanceCovered = $currentOdometer - $prevLog['odometer_reading'];
                    $fuelEfficiency = $distanceCovered / $data['fuel_quantity'];
                }
                
                $stmt = $conn->prepare("
                    INSERT INTO fuel_logs (vehicle_id, log_date, odometer_reading, fuel_quantity, fuel_cost, 
                        fuel_type, station_name, distance_covered, fuel_efficiency, filled_by, receipt_number, notes)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $data['vehicle_id'],
                    $data['log_date'] ?? date('Y-m-d'),
                    $currentOdometer,
                    $data['fuel_quantity'],
                    $data['fuel_cost'],
                    $data['fuel_type'] ?? 'diesel',
                    $data['station_name'] ?? null,
                    $distanceCovered,
                    $fuelEfficiency,
                    $data['filled_by'] ?? null,
                    $data['receipt_number'] ?? null,
                    $data['notes'] ?? null
                ]);
                
                $logId = $conn->lastInsertId();
                $stmt = $conn->prepare("SELECT * FROM fuel_logs WHERE fuel_log_id = ?");
                $stmt->execute([$logId]);
                
                http_response_code(201);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;

            case 'PUT':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Fuel log ID is required']);
                    return;
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                $updates = [];
                $params = [];
                $allowedFields = ['log_date', 'odometer_reading', 'fuel_quantity', 'fuel_cost', 'fuel_type', 
                    'station_name', 'filled_by', 'receipt_number', 'notes'];
                
                foreach ($data as $key => $value) {
                    if (in_array($key, $allowedFields)) {
                        $updates[] = "$key = ?";
                        $params[] = $value;
                    }
                }
                
                if (empty($updates)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'No valid fields to update']);
                    return;
                }
                
                $params[] = $id;
                $stmt = $conn->prepare("UPDATE fuel_logs SET " . implode(', ', $updates) . " WHERE fuel_log_id = ?");
                $stmt->execute($params);
                
                $stmt = $conn->prepare("SELECT * FROM fuel_logs WHERE fuel_log_id = ?");
                $stmt->execute([$id]);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;
                
            case 'DELETE':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Fuel log ID is required']);
                    return;
                }
                
                $stmt = $conn->prepare("DELETE FROM fuel_logs WHERE fuel_log_id = ?");
                $stmt->execute([$id]);
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true]);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Fuel log not found']);
                }
                break;
                
            default:
                http_response_code(405);
                echo json_encode(['error' => 'Method not allowed']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}
