<?php

function handleSubjects($conn, $method, $data = null) {
    // Log incoming request
    error_log("Handling subjects request: " . $method);
    
    // Handle preflight requests
    if ($method === 'OPTIONS') {
        return ['success' => true];
    }

    try {
        // Create subjects table if it doesn't exist
        $conn->exec("CREATE TABLE IF NOT EXISTS subjects (
            id INT AUTO_INCREMENT PRIMARY KEY,
            subject_name VARCHAR(100) NOT NULL,
            subject_code VARCHAR(20),
            description TEXT,
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        )");

        switch($method) {
            case 'GET':
                // Build query
                $query = "SELECT * FROM subjects WHERE is_active = 1 ORDER BY subject_name ASC";
                $stmt = $conn->prepare($query);
                $stmt->execute();
                $subjects = $stmt->fetchAll();
                
                return ['success' => true, 'data' => $subjects];

            case 'POST':
                try {
                    // Use provided data or parse from input
                    if (!$data) {
                        $data = json_decode(file_get_contents('php://input'), true);
                    }
                    
                    $stmt = $conn->prepare("INSERT INTO subjects (subject_name, subject_code, description) VALUES (?, ?, ?)");
                    $stmt->execute([
                        $data['subject_name'] ?? $data['name'] ?? '',
                        $data['subject_code'] ?? '',
                        $data['description'] ?? ''
                    ]);
                    
                    return ['success' => true, 'message' => 'Subject created', 'id' => $conn->lastInsertId()];
                } catch (PDOException $e) {
                    return ['success' => false, 'error' => 'Failed to create subject: ' . $e->getMessage()];
                }

            case 'PUT':
                try {
                    // Use provided data or parse from input
                    if (!$data) {
                        $data = json_decode(file_get_contents('php://input'), true);
                    }
                    
                    $stmt = $conn->prepare("UPDATE subjects SET subject_name = ?, subject_code = ?, description = ? WHERE id = ?");
                    $stmt->execute([
                        $data['subject_name'] ?? $data['name'] ?? '',
                        $data['subject_code'] ?? '',
                        $data['description'] ?? '',
                        $data['id']
                    ]);
                    
                    return ['success' => true, 'message' => 'Subject updated'];
                } catch (PDOException $e) {
                    return ['success' => false, 'error' => 'Failed to update subject: ' . $e->getMessage()];
                }

            case 'DELETE':
                try {
                    $id = $data['id'] ?? $_GET['id'] ?? null;
                    if (!$id) {
                        throw new Exception('Subject ID is required');
                    }
                    $stmt = $conn->prepare("UPDATE subjects SET is_active = 0 WHERE id = ?");
                    $stmt->execute([$id]);
                    
                    return ['success' => true, 'message' => 'Subject deleted'];
                } catch (Exception $e) {
                    return ['success' => false, 'error' => $e->getMessage()];
                }

            default:
                return ['success' => false, 'error' => 'Method not allowed'];
        }
    } catch (Exception $e) {
        error_log("Subjects handler error: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}