<?php
/**
 * Transport System API Handlers
 * Handles routes, vehicles, students, attendance, payments, and fuel logs
 */

/**
 * Handle transport routes
 */
function handleTransportRoutes($conn, $method) {
    $id = $_GET['id'] ?? null;
    
    try {
        switch($method) {
            case 'GET':
                if ($id) {
                    // Get single route with stops
                    $stmt = $conn->prepare("
                        SELECT r.*, 
                               (SELECT COUNT(*) FROM route_stops WHERE route_id = r.route_id) as stop_count,
                               (SELECT COUNT(*) FROM student_transport WHERE route_id = r.route_id AND status = 'active') as student_count
                        FROM transport_routes r 
                        WHERE r.route_id = ?
                    ");
                    $stmt->execute([$id]);
                    $route = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($route) {
                        // Get stops for this route
                        $stopsStmt = $conn->prepare("SELECT * FROM route_stops WHERE route_id = ? ORDER BY stop_order");
                        $stopsStmt->execute([$id]);
                        $route['stops'] = $stopsStmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        echo json_encode($route);
                    } else {
                        http_response_code(404);
                        echo json_encode(['error' => 'Route not found']);
                    }
                } else {
                    // Get all routes
                    $stmt = $conn->prepare("
                        SELECT r.*, 
                               (SELECT COUNT(*) FROM route_stops WHERE route_id = r.route_id) as stop_count,
                               (SELECT COUNT(*) FROM student_transport WHERE route_id = r.route_id AND status = 'active') as student_count
                        FROM transport_routes r 
                        ORDER BY r.route_number
                    ");
                    $stmt->execute();
                    $routes = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    echo json_encode($routes);
                }
                break;

            case 'POST':
                $data = json_decode(file_get_contents('php://input'), true);
                
                // Validate required fields
                if (empty($data['route_name']) || empty($data['route_number'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Route name and number are required']);
                    return;
                }
                
                $stmt = $conn->prepare("
                    INSERT INTO transport_routes (route_name, route_number, description, total_distance, estimated_duration, base_fee, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $data['route_name'],
                    $data['route_number'],
                    $data['description'] ?? null,
                    $data['total_distance'] ?? 0,
                    $data['estimated_duration'] ?? 0,
                    $data['base_fee'] ?? 0,
                    $data['status'] ?? 'active'
                ]);
                
                $routeId = $conn->lastInsertId();
                
                // Get created route
                $stmt = $conn->prepare("SELECT * FROM transport_routes WHERE route_id = ?");
                $stmt->execute([$routeId]);
                $route = $stmt->fetch(PDO::FETCH_ASSOC);
                
                http_response_code(201);
                echo json_encode($route);
                break;

            case 'PUT':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Route ID is required']);
                    return;
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                $updates = [];
                $params = [];
                $allowedFields = ['route_name', 'route_number', 'description', 'total_distance', 'estimated_duration', 'base_fee', 'status'];
                
                foreach ($data as $key => $value) {
                    if (in_array($key, $allowedFields)) {
                        $updates[] = "$key = ?";
                        $params[] = $value;
                    }
                }
                
                if (empty($updates)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'No valid fields to update']);
                    return;
                }
                
                $params[] = $id;
                $sql = "UPDATE transport_routes SET " . implode(', ', $updates) . " WHERE route_id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->execute($params);
                
                // Return updated route
                $stmt = $conn->prepare("SELECT * FROM transport_routes WHERE route_id = ?");
                $stmt->execute([$id]);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;

            case 'DELETE':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Route ID is required']);
                    return;
                }
                
                $stmt = $conn->prepare("DELETE FROM transport_routes WHERE route_id = ?");
                $stmt->execute([$id]);
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true]);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Route not found']);
                }
                break;
                
            default:
                http_response_code(405);
                echo json_encode(['error' => 'Method not allowed']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

/**
 * Handle transport vehicles
 */
function handleTransportVehicles($conn, $method) {
    $id = $_GET['id'] ?? null;
    
    try {
        switch($method) {
            case 'GET':
                if ($id) {
                    $stmt = $conn->prepare("SELECT * FROM transport_vehicles WHERE vehicle_id = ?");
                    $stmt->execute([$id]);
                    $vehicle = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($vehicle) {
                        echo json_encode($vehicle);
                    } else {
                        http_response_code(404);
                        echo json_encode(['error' => 'Vehicle not found']);
                    }
                } else {
                    $stmt = $conn->prepare("SELECT * FROM transport_vehicles ORDER BY vehicle_number");
                    $stmt->execute();
                    echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
                }
                break;

            case 'POST':
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (empty($data['vehicle_number']) || empty($data['capacity'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Vehicle number and capacity are required']);
                    return;
                }
                
                $stmt = $conn->prepare("
                    INSERT INTO transport_vehicles (vehicle_number, vehicle_type, capacity, model, year, registration_number, 
                        fuel_type, average_fuel_consumption, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $data['vehicle_number'],
                    $data['vehicle_type'] ?? 'bus',
                    $data['capacity'],
                    $data['model'] ?? null,
                    $data['year'] ?? null,
                    $data['registration_number'] ?? null,
                    $data['fuel_type'] ?? 'diesel',
                    $data['average_fuel_consumption'] ?? null,
                    $data['status'] ?? 'active'
                ]);
                
                $vehicleId = $conn->lastInsertId();
                $stmt = $conn->prepare("SELECT * FROM transport_vehicles WHERE vehicle_id = ?");
                $stmt->execute([$vehicleId]);
                
                http_response_code(201);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;

            case 'PUT':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Vehicle ID is required']);
                    return;
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                $updates = [];
                $params = [];
                $allowedFields = ['vehicle_number', 'vehicle_type', 'capacity', 'model', 'year', 'registration_number', 
                    'insurance_expiry', 'license_expiry', 'fuel_type', 'average_fuel_consumption', 'status'];
                
                foreach ($data as $key => $value) {
                    if (in_array($key, $allowedFields)) {
                        $updates[] = "$key = ?";
                        $params[] = $value;
                    }
                }
                
                if (empty($updates)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'No valid fields to update']);
                    return;
                }
                
                $params[] = $id;
                $stmt = $conn->prepare("UPDATE transport_vehicles SET " . implode(', ', $updates) . " WHERE vehicle_id = ?");
                $stmt->execute($params);
                
                $stmt = $conn->prepare("SELECT * FROM transport_vehicles WHERE vehicle_id = ?");
                $stmt->execute([$id]);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;

            case 'DELETE':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Vehicle ID is required']);
                    return;
                }
                
                $stmt = $conn->prepare("DELETE FROM transport_vehicles WHERE vehicle_id = ?");
                $stmt->execute([$id]);
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true]);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Vehicle not found']);
                }
                break;
                
            default:
                http_response_code(405);
                echo json_encode(['error' => 'Method not allowed']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

/**
 * Handle student transport registrations
 */
function handleTransportStudents($conn, $method) {
    $id = $_GET['id'] ?? null;
    
    try {
        switch($method) {
            case 'GET':
                if ($id) {
                    $stmt = $conn->prepare("
                        SELECT st.*, s.first_name, s.last_name, s.class, r.route_name, rs.stop_name
                        FROM student_transport st
                        JOIN students s ON st.student_id = s.student_id
                        JOIN transport_routes r ON st.route_id = r.route_id
                        LEFT JOIN route_stops rs ON st.stop_id = rs.stop_id
                        WHERE st.transport_registration_id = ?
                    ");
                    $stmt->execute([$id]);
                    $registration = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($registration) {
                        echo json_encode($registration);
                    } else {
                        http_response_code(404);
                        echo json_encode(['error' => 'Registration not found']);
                    }
                } else {
                    $stmt = $conn->prepare("
                        SELECT st.*, 
                               CONCAT(s.first_name, ' ', s.last_name) as student_name,
                               s.class, r.route_name, rs.stop_name
                        FROM student_transport st
                        JOIN students s ON st.student_id = s.student_id
                        JOIN transport_routes r ON st.route_id = r.route_id
                        LEFT JOIN route_stops rs ON st.stop_id = rs.stop_id
                        ORDER BY st.created_at DESC
                    ");
                    $stmt->execute();
                    echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
                }
                break;

            case 'POST':
                $data = json_decode(file_get_contents('php://input'), true);
                
                if (empty($data['student_id']) || empty($data['route_id']) || empty($data['stop_id'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Student ID, route ID, and stop ID are required']);
                    return;
                }
                
                $stmt = $conn->prepare("
                    INSERT INTO student_transport (student_id, route_id, stop_id, transport_fee, discount_percentage, 
                        final_fee, billing_cycle, start_date, status)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $transportFee = $data['transport_fee'] ?? 0;
                $discount = $data['discount_percentage'] ?? 0;
                $finalFee = $transportFee - ($transportFee * $discount / 100);
                
                $stmt->execute([
                    $data['student_id'],
                    $data['route_id'],
                    $data['stop_id'],
                    $transportFee,
                    $discount,
                    $finalFee,
                    $data['billing_cycle'] ?? 'termly',
                    $data['start_date'] ?? date('Y-m-d'),
                    $data['status'] ?? 'active'
                ]);
                
                $regId = $conn->lastInsertId();
                $stmt = $conn->prepare("
                    SELECT st.*, CONCAT(s.first_name, ' ', s.last_name) as student_name
                    FROM student_transport st
                    JOIN students s ON st.student_id = s.student_id
                    WHERE st.transport_registration_id = ?
                ");
                $stmt->execute([$regId]);
                
                http_response_code(201);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;

            case 'PUT':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Registration ID is required']);
                    return;
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                $updates = [];
                $params = [];
                $allowedFields = ['route_id', 'stop_id', 'transport_fee', 'discount_percentage', 'final_fee', 
                    'billing_cycle', 'status', 'special_requirements'];
                
                foreach ($data as $key => $value) {
                    if (in_array($key, $allowedFields)) {
                        $updates[] = "$key = ?";
                        $params[] = $value;
                    }
                }
                
                if (empty($updates)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'No valid fields to update']);
                    return;
                }
                
                $params[] = $id;
                $stmt = $conn->prepare("UPDATE student_transport SET " . implode(', ', $updates) . " WHERE transport_registration_id = ?");
                $stmt->execute($params);
                
                $stmt = $conn->prepare("SELECT * FROM student_transport WHERE transport_registration_id = ?");
                $stmt->execute([$id]);
                echo json_encode($stmt->fetch(PDO::FETCH_ASSOC));
                break;
                
            case 'DELETE':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Registration ID is required']);
                    return;
                }
                
                $stmt = $conn->prepare("DELETE FROM student_transport WHERE transport_registration_id = ?");
                $stmt->execute([$id]);
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true]);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Registration not found']);
                }
                break;
                
            default:
                http_response_code(405);
                echo json_encode(['error' => 'Method not allowed']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}
