<?php
// Suppress all PHP errors from being displayed as HTML
error_reporting(E_ALL);
ini_set('display_errors', '0');
ini_set('display_startup_errors', '0');

// Global error handler to return JSON instead of HTML
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    error_log("PHP Error [$errno]: $errstr in $errfile on line $errline");
    if (!headers_sent()) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
    }
    echo json_encode([
        'success' => false,
        'error' => 'Internal server error',
        'message' => $errstr,
        'file' => basename($errfile),
        'line' => $errline
    ]);
    exit;
});

// Global exception handler
set_exception_handler(function($exception) {
    error_log("Uncaught Exception: " . $exception->getMessage());
    if (!headers_sent()) {
        header('Content-Type: application/json; charset=utf-8');
        http_response_code(500);
    }
    echo json_encode([
        'success' => false,
        'error' => 'Internal server error',
        'message' => $exception->getMessage()
    ]);
    exit;
});

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/middleware/cors.php';

// Handle CORS
handleCORS();

// Set content type
header('Content-Type: application/json; charset=utf-8');

// JWT Secret Key - Define at the top before any functions use it
define('JWT_SECRET', 'your_jwt_secret_key_here_change_in_production');
define('JWT_ALGORITHM', 'HS256');

// Include handler files (only if they exist)
if (file_exists(__DIR__ . '/handlers/subjects.php')) {
    require_once __DIR__ . '/handlers/subjects.php';
}
if (file_exists(__DIR__ . '/handlers/scores.php')) {
    require_once __DIR__ . '/handlers/scores.php';
}
if (file_exists(__DIR__ . '/handlers/transport.php')) {
    require_once __DIR__ . '/handlers/transport.php';
}
if (file_exists(__DIR__ . '/handlers/transport_attendance.php')) {
    require_once __DIR__ . '/handlers/transport_attendance.php';
}
if (file_exists(__DIR__ . '/handlers/transport_payments.php')) {
    require_once __DIR__ . '/handlers/transport_payments.php';
}
if (file_exists(__DIR__ . '/handlers/fuel_logs.php')) {
    require_once __DIR__ . '/handlers/fuel_logs.php';
}
// Include academic handler
if (file_exists(__DIR__ . '/handlers/academic.php')) {
    require_once __DIR__ . '/handlers/academic.php';
}
// Include assessments_extended handler (function conflicts resolved)
if (file_exists(__DIR__ . '/handlers/assessments_extended.php')) {
    require_once __DIR__ . '/handlers/assessments_extended.php';
}
// Include SMS handler
if (file_exists(__DIR__ . '/handlers/sms.php')) {
    require_once __DIR__ . '/handlers/sms.php';
}

// Database configuration - Auto-detect environment
$isProduction = isset($_SERVER['HTTP_HOST']) && (
    strpos($_SERVER['HTTP_HOST'], 'ceiscy.com') !== false ||
    strpos($_SERVER['HTTP_HOST'], 'dgsch.vercel.app') !== false ||
    strpos($_SERVER['HTTP_HOST'], 'pcg.campus-plug.com') !== false
);

if ($isProduction) {
    // Production database (StormerHost)
    $servername = "localhost";
    $username = "campuspl_pcg";
    $password = "pcgcctt@30";
    $dbname = "campuspl_dgsch";
} else {
    // Local development database - XAMPP
    $servername = "localhost";
    $username = "root";
    $password = "";
    $dbname = "divine_grace_school";
}

try {
    // Create connection with error handling
    try {
        $conn = new PDO("mysql:host=$servername;dbname=$dbname;charset=utf8mb4", $username, $password);
        $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // If database doesn't exist, try to create it
        if ($e->getCode() == 1049) { // Database doesn't exist
            $conn = new PDO("mysql:host=$servername", $username, $password);
            $conn->exec("CREATE DATABASE IF NOT EXISTS `$dbname` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
            $conn->exec("USE `$dbname`");
            // You might want to execute your schema here or redirect to setup
            header('Location: /DIVINE2/database/init_database.php');
            exit;
        } else {
            throw $e; // Re-throw other PDO exceptions
        }
    }

    // Ensure required tables exist
    try {
        ensureFeedingTables($conn);
    } catch (Exception $e) {
        // Log error but don't crash - tables might already exist
        error_log("ensureFeedingTables error: " . $e->getMessage());
    }

    $method = $_SERVER['REQUEST_METHOD'];
    $request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);

// Determine the actual base path dynamically (e.g. /DIVINE2/api, /school/api, /api)
$scriptDir = str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? ''));
$basePath = rtrim($scriptDir, '/');

$trimmedPath = $request_uri;
if (!empty($basePath) && strpos($request_uri, $basePath) === 0) {
    $trimmedPath = substr($request_uri, strlen($basePath));
}
$endpointPath = $trimmedPath;

// Normalize path and extract segments
$pathSegments = array_values(array_filter(explode('/', trim($endpointPath, '/')), function($segment) {
    return $segment !== '' && $segment !== 'index.php';
}));

// If the first segment is "api", drop it (shouldn't happen with query param)
if (!empty($pathSegments) && strtolower($pathSegments[0]) === 'api') {
    array_shift($pathSegments);
}

// Determine endpoint - handle multi-part endpoints like "feeding-records/clear-all"
$endpoint = $pathSegments[0] ?? '';

// For debugging: log the path segments
error_log("Path segments: " . json_encode($pathSegments));

    switch($endpoint) {
        case '':
        case 'test':
            // Simple test endpoint (no auth required)
            echo json_encode([
                'success' => true,
                'message' => 'API is working!',
                'timestamp' => date('Y-m-d H:i:s'),
                'endpoint' => $endpoint,
                'method' => $method
            ]);
            break;
            
        case 'students':
            handleStudents($conn, $method);
            break;
        case 'subjects':
            if (function_exists('handleSubjects')) {
                handleSubjects($conn, $method);
            } else {
                http_response_code(501);
                echo json_encode(['error' => 'Subjects endpoint not implemented']);
            }
            break;
        case 'scores':
            if (function_exists('handleScores')) {
                handleScores($conn, $method);
            } else {
                http_response_code(501);
                echo json_encode(['error' => 'Scores endpoint not implemented']);
            }
            break;
        case 'fees':
            handleFees($conn, $method);
            break;
        case 'expenses':
            handleExpenses($conn, $method);
            break;
        case 'income':
            handleIncome($conn, $method);
            break;
        case 'academics':
            handleAcademics($conn, $method);
            break;
        case 'auth':
            handleAuth($conn, $method);
            break;
        case 'register':
            handleRegister($conn, $method);
            break;
        case 'reports':
            handleReports($conn, $method);
            break;
        case 'feeding':
            // Check if this is a clear-all action
            if ($method === 'DELETE' && isset($_GET['action']) && $_GET['action'] === 'clear-all') {
                handleClearAllFeedingRecords($conn, $method);
            } else {
                handleFeeding($conn, $method);
            }
            break;
        case 'feeding-fees':
            handleFeedingFees($conn, $method);
            break;
        case 'feeding-stats':
            handleFeedingStats($conn, $method);
            break;
        case 'feeding-class-details':
            handleFeedingClassDetails($conn, $method);
            break;
        case 'feeding-student-settings':
            handleFeedingStudentSettings($conn, $method);
            break;
        
        // Transport System Endpoints
        case 'transport-routes':
            handleTransportRoutes($conn, $method);
            break;
        case 'transport-vehicles':
            handleTransportVehicles($conn, $method);
            break;
        case 'transport-students':
            handleTransportStudents($conn, $method);
            break;
        case 'transport-attendance':
            handleTransportAttendance($conn, $method);
            break;
        case 'transport-payments':
            handleTransportPayments($conn, $method);
            break;
        case 'fuel-logs':
            handleFuelLogs($conn, $method);
            break;
        case 'sms':
            if (function_exists('handleSMS')) {
                handleSMS($conn, $method);
            } else {
                http_response_code(501);
                echo json_encode(['error' => 'SMS endpoint not implemented']);
            }
            break;

        default:
            http_response_code(404);
            echo json_encode(['error' => 'Endpoint not found']);
    }

} catch(PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
}

/**
 * Handle student-related requests
 */
function handleStudents($conn, $method) {
    $user = requireAuth(); // Require authentication for all student endpoints
    $id = $_GET['id'] ?? null;
    
    try {
        switch($method) {
            case 'GET':
                if ($id) {
                    // Get single student by ID with additional details
                    $stmt = $conn->prepare("
                        SELECT s.*, 
                               (SELECT COUNT(*) FROM fees f WHERE f.student_id = s.id) as fee_count,
                               (SELECT SUM(amount) FROM fees f WHERE f.student_id = s.id AND f.status = 'Paid') as total_paid,
                               (SELECT SUM(amount) FROM fees f WHERE f.student_id = s.id AND f.status != 'Paid') as total_pending
                        FROM students s 
                        WHERE s.id = ?
                    ");
                    $stmt->execute([$id]);
                    $student = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($student) {
                        // Get student's fees if requested
                        if (isset($_GET['include']) && $_GET['include'] === 'fees') {
                            $feesStmt = $conn->prepare("SELECT * FROM fees WHERE student_id = ? ORDER BY due_date DESC");
                            $feesStmt->execute([$id]);
                            $student['fees'] = $feesStmt->fetchAll(PDO::FETCH_ASSOC);
                        }
                        
                        // Map to camelCase for frontend compatibility
                        $mappedStudent = [
                            'id' => $student['id'],
                            'studentId' => $student['student_id'],
                            'name' => trim($student['first_name'] . ' ' . $student['last_name']),
                            'class' => $student['class'],
                            'gender' => $student['gender'],
                            'dateOfBirth' => $student['date_of_birth'] ?? null,
                            'parentName' => $student['father_name'] ?? $student['parent_name'] ?? null,
                            'parentContact' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                            'parentPhone' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                            'parentEmail' => $student['parent_email'] ?? null,
                            'address' => $student['address'] ?? null,
                            'status' => $student['status'] ?? 'active',
                            'enrollmentDate' => $student['enrollment_date'] ?? null,
                            'createdAt' => $student['created_at'],
                            'updatedAt' => $student['updated_at'],
                            'feeCount' => $student['fee_count'] ?? 0,
                            'totalPaid' => $student['total_paid'] ?? 0,
                            'totalPending' => $student['total_pending'] ?? 0
                        ];
                        
                        if (isset($student['fees'])) {
                            $mappedStudent['fees'] = $student['fees'];
                        }
                        
                        echo json_encode($mappedStudent);
                    } else {
                        http_response_code(404);
                        echo json_encode(['error' => 'Student not found']);
                    }
                } else {
                    // Get all students with pagination and filtering
                    $page = max(1, intval($_GET['page'] ?? 1));
                    $limit = min(10000, max(1, intval($_GET['limit'] ?? 20)));
                    $offset = ($page - 1) * $limit;
                    
                    // Build query based on filters
                    $where = [];
                    $params = [];
                    
                    // Search filter
                    if (!empty($_GET['search'])) {
                        $search = "%{$_GET['search']}%";
                        $where[] = "(first_name LIKE ? OR last_name LIKE ? OR student_id LIKE ? OR phone_number LIKE ?)";
                        $params = array_merge($params, [$search, $search, $search, $search]);
                    }
                    
                    // Class filter
                    if (!empty($_GET['class'])) {
                        $where[] = "class = ?";
                        $params[] = $_GET['class'];
                    }
                    
                    // Status filter
                    if (!empty($_GET['status'])) {
                        $where[] = "status = ?";
                        $params[] = $_GET['status'];
                    }
                    
                    $whereClause = $where ? 'WHERE ' . implode(' AND ', $where) : '';
                    
                    // Get total count for pagination
                    $countStmt = $conn->prepare("SELECT COUNT(*) as total FROM students $whereClause");
                    $countStmt->execute($params);
                    $total = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
                    $totalPages = ceil($total / $limit);
                    
                    // Get paginated results
                    // Note: LIMIT and OFFSET must be integers, not bound parameters in some MySQL versions
                    $stmt = $conn->prepare("
                        SELECT s.*, 
                               (SELECT COUNT(*) FROM fees f WHERE f.student_id = s.id) as fee_count,
                               (SELECT SUM(amount) FROM fees f WHERE f.student_id = s.id AND f.status = 'Paid') as total_paid
                        FROM students s 
                        $whereClause 
                        ORDER BY s.created_at DESC 
                        LIMIT $limit OFFSET $offset
                    ");
                    $stmt->execute($params);
                    $students = $stmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    // Map snake_case to camelCase for frontend compatibility
                    $students = array_map(function($student) {
                        return [
                            'id' => $student['id'],
                            'studentId' => $student['student_id'],
                            'name' => trim($student['first_name'] . ' ' . $student['last_name']),
                            'class' => $student['class'],
                            'gender' => $student['gender'],
                            'dateOfBirth' => $student['date_of_birth'] ?? null,
                            'parentName' => $student['father_name'] ?? $student['parent_name'] ?? null,
                            'parentContact' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                            'parentPhone' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                            'parentEmail' => $student['parent_email'] ?? null,
                            'address' => $student['address'] ?? null,
                            'status' => $student['status'] ?? 'active',
                            'enrollmentDate' => $student['enrollment_date'] ?? null,
                            'createdAt' => $student['created_at'],
                            'updatedAt' => $student['updated_at'],
                            'feeCount' => $student['fee_count'] ?? 0,
                            'totalPaid' => $student['total_paid'] ?? 0
                        ];
                    }, $students);
                    
                    echo json_encode([
                        'data' => $students,
                        'pagination' => [
                            'total' => (int)$total,
                            'page' => $page,
                            'limit' => $limit,
                            'total_pages' => $totalPages
                        ]
                    ]);
                }
                break;
                
            case 'POST':
                $data = json_decode(file_get_contents('php://input'), true);
                
                // Validate required fields (date_of_birth is required)
                $required = ['name', 'class', 'gender', 'dateOfBirth'];
                $missing = [];
                foreach ($required as $field) {
                    if (empty($data[$field])) {
                        $missing[] = $field;
                    }
                }
                
                if (!empty($missing)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Missing required fields: ' . implode(', ', $missing)]);
                    return;
                }
                
                // Generate student ID if not provided
                if (empty($data['student_id'])) {
                    $year = date('y');
                    $random = strtoupper(substr(md5(uniqid()), 0, 4));
                    $data['student_id'] = "STD{$year}{$random}";
                }
                
                // Parse name into first_name and last_name
                $nameParts = explode(' ', trim($data['name']), 2);
                $firstName = $nameParts[0];
                $lastName = $nameParts[1] ?? '';
                
                // Generate student ID if not provided or if it already exists
                $studentId = $data['studentId'] ?? $data['student_id'] ?? null;
                
                // Check if student ID already exists
                if ($studentId) {
                    $checkStmt = $conn->prepare("SELECT id FROM students WHERE student_id = ?");
                    $checkStmt->execute([$studentId]);
                    if ($checkStmt->fetch()) {
                        // ID exists, generate a new one
                        $studentId = null;
                    }
                }
                
                // Auto-generate student ID if needed
                if (!$studentId) {
                    // Find the highest DG### number
                    $stmt = $conn->prepare("SELECT student_id FROM students WHERE student_id LIKE 'DG%' ORDER BY student_id DESC LIMIT 1");
                    $stmt->execute();
                    $lastStudent = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($lastStudent) {
                        $lastId = $lastStudent['student_id'];
                        $number = (int)substr($lastId, 2); // Remove 'DG' prefix
                        $nextNumber = $number + 1;
                    } else {
                        $nextNumber = 1;
                    }
                    
                    $studentId = 'DG' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);
                }
                
                // Insert student
                $stmt = $conn->prepare("
                    INSERT INTO students (
                        student_id, first_name, last_name, class, date_of_birth, gender, 
                        father_name, phone_number, address, section, admission_date, status, created_at
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                ");
                
                $status = $data['status'] ?? 'Active';
                $section = $data['section'] ?? 'A';
                $admissionDate = date('Y-m-d');
                
                $stmt->execute([
                    $studentId,
                    $firstName,
                    $lastName,
                    $data['class'],
                    $data['dateOfBirth'] ?? $data['date_of_birth'],
                    $data['gender'],
                    $data['parentName'] ?? $data['parent_name'] ?? $data['father_name'] ?? null,
                    $data['parentContact'] ?? $data['parent_phone'] ?? $data['parentPhone'] ?? $data['phone_number'] ?? null,
                    $data['address'] ?? null,
                    $section,
                    $admissionDate,
                    $status
                ]);
                
                $studentId = $conn->lastInsertId();
                
                // Return created student data
                $stmt = $conn->prepare("SELECT * FROM students WHERE id = ?");
                $stmt->execute([$studentId]);
                $student = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Map to camelCase
                $student = [
                    'id' => $student['id'],
                    'studentId' => $student['student_id'],
                    'name' => trim($student['first_name'] . ' ' . $student['last_name']),
                    'class' => $student['class'],
                    'gender' => $student['gender'],
                    'dateOfBirth' => $student['date_of_birth'] ?? null,
                    'parentName' => $student['father_name'] ?? $student['parent_name'] ?? null,
                    'parentContact' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                    'parentPhone' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                    'parentEmail' => $student['parent_email'] ?? null,
                    'address' => $student['address'] ?? null,
                    'status' => $student['status'] ?? 'active',
                    'enrollmentDate' => $student['enrollment_date'] ?? null,
                    'createdAt' => $student['created_at'],
                    'updatedAt' => $student['updated_at']
                ];
                
                http_response_code(201);
                echo json_encode($student);
                break;
                
            case 'PUT':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Student ID is required']);
                    return;
                }
                
                $data = json_decode(file_get_contents('php://input'), true);
                
                // Build update query
                $updates = [];
                $params = [];
                $allowedFields = [
                    'first_name', 'last_name', 'class', 'date_of_birth', 'gender', 
                    'parent_name', 'father_name', 'parent_phone', 'phone_number', 'address', 'status'
                ];
                
                // Map camelCase to snake_case
                $fieldMap = [
                    'parentName' => 'father_name',
                    'parentContact' => 'phone_number',
                    'parentPhone' => 'phone_number',
                    'dateOfBirth' => 'date_of_birth',
                    'firstName' => 'first_name',
                    'lastName' => 'last_name'
                ];
                
                // Handle 'name' field - split into first_name and last_name
                if (isset($data['name'])) {
                    $nameParts = explode(' ', trim($data['name']), 2);
                    $data['first_name'] = $nameParts[0];
                    $data['last_name'] = $nameParts[1] ?? '';
                    unset($data['name']);
                }
                
                foreach ($data as $key => $value) {
                    // Skip nested objects/arrays (like fees object)
                    if (is_array($value) || is_object($value)) {
                        error_log("Skipping nested field: $key");
                        continue;
                    }
                    
                    $dbKey = $fieldMap[$key] ?? $key;
                    error_log("Processing field: $key => $dbKey (value: $value)");
                    
                    if (in_array($dbKey, $allowedFields)) {
                        $updates[] = "$dbKey = ?";
                        $params[] = $value;
                        error_log("✅ Added to update: $dbKey = $value");
                    } else {
                        error_log("❌ Skipped (not allowed): $dbKey");
                    }
                }
                
                if (empty($updates)) {
                    http_response_code(400);
                    echo json_encode(['error' => 'No valid fields to update']);
                    return;
                }
                
                // First check if student exists
                $checkStmt = $conn->prepare("SELECT id FROM students WHERE id = ?");
                $checkStmt->execute([$id]);
                if (!$checkStmt->fetch()) {
                    http_response_code(404);
                    echo json_encode(['error' => 'Student not found']);
                    return;
                }
                
                $params[] = $id; // Add ID for WHERE clause
                
                $sql = "UPDATE students SET " . implode(', ', $updates) . " WHERE id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->execute($params);
                
                // Return updated student data (even if no changes were made)
                $stmt = $conn->prepare("
                    SELECT s.*, 
                           (SELECT COUNT(*) FROM fees f WHERE f.student_id = s.id) as fee_count,
                           (SELECT SUM(amount) FROM fees f WHERE f.student_id = s.id AND f.status = 'Paid') as total_paid
                    FROM students s 
                    WHERE s.id = ?
                ");
                $stmt->execute([$id]);
                $student = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Map to camelCase
                $student = [
                    'id' => $student['id'],
                    'studentId' => $student['student_id'],
                    'name' => trim($student['first_name'] . ' ' . $student['last_name']),
                    'class' => $student['class'],
                    'gender' => $student['gender'],
                    'dateOfBirth' => $student['date_of_birth'] ?? null,
                    'parentName' => $student['father_name'] ?? $student['parent_name'] ?? null,
                    'parentContact' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                    'parentPhone' => $student['phone_number'] ?? $student['parent_phone'] ?? null,
                    'parentEmail' => $student['parent_email'] ?? null,
                    'address' => $student['address'] ?? null,
                    'status' => $student['status'] ?? 'active',
                    'enrollmentDate' => $student['enrollment_date'] ?? null,
                    'createdAt' => $student['created_at'],
                    'updatedAt' => $student['updated_at'],
                    'feeCount' => $student['fee_count'] ?? 0,
                    'totalPaid' => $student['total_paid'] ?? 0
                ];
                
                echo json_encode($student);
                break;
                
            case 'DELETE':
                if (!$id) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Student ID is required']);
                    return;
                }
                
                // Check if student has any fees or payments
                $checkStmt = $conn->prepare("
                    SELECT 
                        (SELECT COUNT(*) FROM fees WHERE student_id = ?) as fee_count
                ");
                $checkStmt->execute([$id]);
                $counts = $checkStmt->fetch(PDO::FETCH_ASSOC);
                
                if ($counts['fee_count'] > 0) {
                    http_response_code(400);
                    echo json_encode([
                        'error' => 'Cannot delete student with associated fees. Mark as inactive instead.',
                        'code' => 'student_has_fees'
                    ]);
                    return;
                }
                
                $stmt = $conn->prepare("DELETE FROM students WHERE id = ?");
                $stmt->execute([$id]);
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true]);
                } else {
                    http_response_code(404);
                    echo json_encode(['error' => 'Student not found']);
                }
                break;
                
            default:
                http_response_code(405);
                echo json_encode(['error' => 'Method not allowed']);
                break;
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

function handleFees($conn, $method) {
    try {
        switch($method) {
            case 'GET':
                $stmt = $conn->prepare("
                    SELECT f.*, 
                           CONCAT(s.first_name, ' ', s.last_name) as studentName, 
                           s.student_id as student_code, 
                           s.class
                    FROM fees f
                    JOIN students s ON f.student_id = s.id
                    ORDER BY f.created_at DESC
                ");
                $stmt->execute();
                $fees = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Map snake_case to camelCase for frontend compatibility
                $fees = array_map(function($fee) {
                    $amount = floatval($fee['amount']);
                    $discount = floatval($fee['discount'] ?? 0);
                    // Use actual amount_paid from database, not status-based calculation
                    $amountPaid = floatval($fee['amount_paid'] ?? 0);
                    $balance = max(0, $amount - $amountPaid);
                    
                    // Determine status based on payment
                    $status = $fee['status'] ?? 'unpaid';
                    if ($amountPaid >= $amount) {
                        $status = 'paid';
                    } elseif ($amountPaid > 0) {
                        $status = 'partial';
                    } else {
                        $status = 'unpaid';
                    }
                    
                    return [
                        'id' => $fee['id'],
                        'studentId' => $fee['student_code'], // Use DG001 format
                        'studentName' => $fee['studentName'],
                        'class' => $fee['class'],
                        'amount' => $amount,
                        'amountPaid' => $amountPaid,
                        'balance' => $balance,
                        'term' => $fee['term'] ?? 'Term 1', // Include term field
                        'description' => $fee['notes'] ?? $fee['fee_type'] ?? 'School Fee',
                        'status' => strtolower($status),
                        'feeType' => $fee['fee_type'] ?? 'Other',
                        'dueDate' => $fee['due_date'] ?? null,
                        'paymentDate' => $fee['payment_date'] ?? null,
                        'paymentMethod' => $fee['payment_method'] ?? null,
                        'discount' => $discount,
                        'scholarshipAmount' => $discount,
                        'paymentHistory' => [],
                        'createdAt' => $fee['created_at'],
                        'updatedAt' => $fee['updated_at']
                    ];
                }, $fees);
                
                echo json_encode($fees);
                break;

        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Handle both camelCase and snake_case field names
            $student_id = $data['student_id'] ?? $data['studentId'] ?? null;
            $student_name = $data['student_name'] ?? $data['studentName'] ?? '';
            $amount = $data['amount'] ?? 0;
            $original_amount = $data['original_amount'] ?? $data['originalAmount'] ?? $amount;
            $scholarship_amount = $data['scholarship_amount'] ?? $data['scholarshipAmount'] ?? 0;
            $description = $data['description'] ?? 'School Fee';
            $date = $data['date'] ?? $data['paymentDate'] ?? $data['dueDate'] ?? date('Y-m-d');
            $status = $data['status'] ?? 'unpaid';
            $type = $data['type'] ?? 'fee';
            
            // Validate required fields
            if (!$student_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Student ID is required']);
                return;
            }
            
            if (!$amount || $amount <= 0) {
                http_response_code(400);
                echo json_encode(['error' => 'Valid amount is required']);
                return;
            }
            
            // Verify student exists - check by both numeric id and student_id (DG001 format)
            $checkStmt = $conn->prepare("SELECT id, CONCAT(first_name, ' ', last_name) as name, student_id FROM students WHERE id = ? OR student_id = ?");
            $checkStmt->execute([$student_id, $student_id]);
            $student = $checkStmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$student) {
                http_response_code(404);
                echo json_encode(['error' => 'Student not found', 'searched_id' => $student_id]);
                return;
            }
            
            // Use the numeric database ID for the foreign key
            $student_db_id = $student['id'];
            
            // Use student name from database if not provided
            if (empty($student_name)) {
                $student_name = $student['name'];
            }
            
            // Map to actual database columns
            $fee_type = $data['feeType'] ?? $data['fee_type'] ?? 'Other';
            $due_date = $data['dueDate'] ?? $data['due_date'] ?? $date;
            $payment_method = $data['paymentMethod'] ?? $data['payment_method'] ?? null;
            $discount = $data['discount'] ?? $data['scholarshipAmount'] ?? 0;
            $notes = $data['notes'] ?? $description;
            
            $stmt = $conn->prepare("INSERT INTO fees (student_id, fee_type, amount, due_date, status, payment_date, payment_method, discount, notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $student_db_id, 
                $fee_type,
                $amount,
                $due_date,
                $status,
                ($status === 'paid' ? $date : null),
                $payment_method,
                $discount,
                $notes
            ]);
            
            $feeId = $conn->lastInsertId();
            
            // Return the created fee with all data
            $stmt = $conn->prepare("
                SELECT f.*, CONCAT(s.first_name, ' ', s.last_name) as studentName, s.student_id, s.class
                FROM fees f
                JOIN students s ON f.student_id = s.id
                WHERE f.id = ?
            ");
            $stmt->execute([$feeId]);
            $newFee = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Map to camelCase
            $newFee = [
                'id' => $newFee['id'],
                'studentId' => $newFee['student_id'],
                'studentName' => $newFee['studentName'],
                'class' => $newFee['class'],
                'amount' => $newFee['amount'],
                'amountPaid' => $newFee['amount_paid'] ?? 0,
                'balance' => $newFee['balance'] ?? $newFee['amount'],
                'description' => $newFee['description'] ?? 'School Fee',
                'status' => $newFee['status'] ?? 'unpaid',
                'term' => $newFee['term'] ?? 'term1',
                'academicYear' => $newFee['academic_year'] ?? date('Y'),
                'dueDate' => $newFee['due_date'] ?? null,
                'paymentDate' => $newFee['payment_date'] ?? null,
                'paymentMethod' => $newFee['payment_method'] ?? null,
                'scholarshipAmount' => $newFee['scholarship_amount'] ?? 0,
                'paymentHistory' => json_decode($newFee['payment_history'] ?? '[]', true),
                'createdAt' => $newFee['created_at'],
                'updatedAt' => $newFee['updated_at']
            ];
            
            echo json_encode($newFee);
            break;

        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Get fee ID
            $fee_id = $data['id'] ?? null;
            if (!$fee_id) {
                http_response_code(400);
                echo json_encode(['error' => 'Fee ID is required']);
                return;
            }
            
            // Get current fee data
            $stmt = $conn->prepare("SELECT * FROM fees WHERE id = ?");
            $stmt->execute([$fee_id]);
            $currentFee = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$currentFee) {
                http_response_code(404);
                echo json_encode(['error' => 'Fee not found']);
                return;
            }
            
            // Handle both camelCase and snake_case field names
            $amount = $data['amount'] ?? $currentFee['amount'];
            $discount = $data['discount'] ?? $data['scholarshipAmount'] ?? $currentFee['discount'];
            $status = $data['status'] ?? $currentFee['status'];
            $description = $data['description'] ?? $data['notes'] ?? $currentFee['notes'];
            $payment_date = $data['payment_date'] ?? $data['paymentDate'] ?? $currentFee['payment_date'];
            $payment_method = $data['payment_method'] ?? $data['paymentMethod'] ?? $currentFee['payment_method'];
            $payment_history = $data['payment_history'] ?? $data['paymentHistory'] ?? null;
            
            // Get amount_paid - frontend already calculates cumulative total
            // So we just use the provided value directly
            $amount_paid = isset($data['amount_paid']) || isset($data['amountPaid']) 
                ? floatval($data['amount_paid'] ?? $data['amountPaid'])
                : floatval($currentFee['amount_paid'] ?? 0);
            
            // If payment_history is an array, encode it as JSON
            if (is_array($payment_history)) {
                $payment_history = json_encode($payment_history);
            }
            
            // CORRECT CALCULATION: balance = original amount - amount paid - discount
            $balance = max(0, $amount - $amount_paid - $discount);
            
            // Update status based on payment
            if ($amount_paid >= $amount) {
                $status = 'Paid';
            } elseif ($amount_paid > 0) {
                $status = 'Partial';
            } else {
                $status = 'Unpaid';
            }
            
            $stmt = $conn->prepare("
                UPDATE fees 
                SET amount = ?, 
                    amount_paid = ?,
                    balance = ?,
                    discount = ?,
                    status = ?, 
                    notes = ?,
                    payment_date = ?,
                    payment_method = ?,
                    updated_at = NOW()
                WHERE id = ?
            ");
            
            $stmt->execute([
                $amount,
                $amount_paid,
                $balance,
                $discount,
                $status, 
                $description,
                $payment_date,
                $payment_method,
                $fee_id
            ]);
            
            // Return the updated fee
            $stmt = $conn->prepare("
                SELECT f.*, CONCAT(s.first_name, ' ', s.last_name) as studentName, s.student_id, s.class
                FROM fees f
                JOIN students s ON f.student_id = s.id
                WHERE f.id = ?
            ");
            $stmt->execute([$fee_id]);
            $updatedFee = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Map to camelCase
            $updatedFee = [
                'id' => $updatedFee['id'],
                'studentId' => $updatedFee['student_id'],
                'studentName' => $updatedFee['studentName'],
                'class' => $updatedFee['class'],
                'amount' => $updatedFee['amount'],
                'amountPaid' => $updatedFee['amount_paid'] ?? 0,
                'balance' => $updatedFee['balance'] ?? $updatedFee['amount'],
                'description' => $updatedFee['description'] ?? 'School Fee',
                'status' => $updatedFee['status'] ?? 'unpaid',
                'term' => $updatedFee['term'] ?? 'term1',
                'academicYear' => $updatedFee['academic_year'] ?? date('Y'),
                'dueDate' => $updatedFee['due_date'] ?? null,
                'paymentDate' => $updatedFee['payment_date'] ?? null,
                'paymentMethod' => $updatedFee['payment_method'] ?? null,
                'scholarshipAmount' => $updatedFee['scholarship_amount'] ?? 0,
                'paymentHistory' => json_decode($updatedFee['payment_history'] ?? '[]', true),
                'createdAt' => $updatedFee['created_at'],
                'updatedAt' => $updatedFee['updated_at']
            ];
            
            echo json_encode($updatedFee);
            break;

        case 'DELETE':
            $id = $_GET['id'] ?? 0;
            $stmt = $conn->prepare("DELETE FROM fees WHERE id=?");
            $stmt->execute([$id]);
            echo json_encode(['success' => true]);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Database error: ' . $e->getMessage(),
            'code' => $e->getCode()
        ]);
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Server error: ' . $e->getMessage()
        ]);
    }
}

function handleExpenses($conn, $method) {
    switch($method) {
        case 'GET':
            $stmt = $conn->prepare("SELECT *, expense_date as date FROM expenses ORDER BY created_at DESC");
            $stmt->execute();
            $expenses = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo json_encode($expenses);
            break;

        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            $description = $data['description'] ?? '';
            $amount = $data['amount'] ?? 0;
            $category = $data['category'] ?? '';
            $date = $data['date'] ?? date('Y-m-d');
            
            $stmt = $conn->prepare("INSERT INTO expenses (description, amount, category, expense_date) VALUES (?, ?, ?, ?)");
            $stmt->execute([$description, $amount, $category, $date]);
            
            $id = $conn->lastInsertId();
            
            // Return the created expense
            $stmt = $conn->prepare("SELECT *, expense_date as date FROM expenses WHERE id = ?");
            $stmt->execute([$id]);
            $expense = $stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode($expense);
            break;

        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            $id = $data['id'] ?? null;
            
            if (!$id) {
                http_response_code(400);
                echo json_encode(['error' => 'ID is required']);
                return;
            }
            
            $stmt = $conn->prepare("UPDATE expenses SET description=?, amount=?, category=?, expense_date=? WHERE id=?");
            $stmt->execute([$data['description'], $data['amount'], $category, $data['date'], $id]);
            
            // Return the updated expense
            $stmt = $conn->prepare("SELECT *, expense_date as date FROM expenses WHERE id = ?");
            $stmt->execute([$id]);
            $expense = $stmt->fetch(PDO::FETCH_ASSOC);
            
            echo json_encode($expense);
            break;

        case 'DELETE':
            $id = $_GET['id'] ?? 0;
            if (!$id) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'Expense ID is required']);
                break;
            }
            
            $stmt = $conn->prepare("DELETE FROM expenses WHERE id=?");
            $result = $stmt->execute([$id]);
            $rowCount = $stmt->rowCount();
            
            if ($rowCount > 0) {
                echo json_encode(['success' => true, 'message' => 'Expense deleted successfully', 'deleted' => $rowCount]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Expense not found or already deleted']);
            }
            break;
    }
}

function handleIncome($conn, $method) {
    switch($method) {
        case 'GET':
            $stmt = $conn->prepare("SELECT *, income_date as date FROM income ORDER BY created_at DESC");
            $stmt->execute();
            $incomes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Map 'source' to 'category' for frontend consistency
            foreach ($incomes as &$income) {
                $income['category'] = $income['source'];
            }
            
            echo json_encode($incomes);
            break;

        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            $description = $data['description'] ?? '';
            $amount = $data['amount'] ?? 0;
            // For income, use category as source if provided, otherwise use default
            $source = $data['category'] ?? $data['source'] ?? 'School Revenue';
            $date = $data['date'] ?? date('Y-m-d');
            
            $stmt = $conn->prepare("INSERT INTO income (description, amount, source, income_date) VALUES (?, ?, ?, ?)");
            $stmt->execute([$description, $amount, $source, $date]);
            
            $id = $conn->lastInsertId();
            
            // Return the created income
            $stmt = $conn->prepare("SELECT *, income_date as date FROM income WHERE id = ?");
            $stmt->execute([$id]);
            $income = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Map 'source' to 'category' for frontend consistency
            $income['category'] = $income['source'];
            
            echo json_encode($income);
            break;

        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            $id = $data['id'] ?? null;
            
            if (!$id) {
                http_response_code(400);
                echo json_encode(['error' => 'ID is required']);
                return;
            }
            
            // Use category as source if provided
            $source = $data['category'] ?? $data['source'] ?? 'School Revenue';
            
            $stmt = $conn->prepare("UPDATE income SET description=?, amount=?, source=?, income_date=? WHERE id=?");
            $stmt->execute([$data['description'], $data['amount'], $source, $data['date'], $id]);
            
            // Return the updated income
            $stmt = $conn->prepare("SELECT *, income_date as date FROM income WHERE id = ?");
            $stmt->execute([$id]);
            $income = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Map 'source' to 'category' for frontend consistency
            $income['category'] = $income['source'];
            
            echo json_encode($income);
            break;

        case 'DELETE':
            $id = $_GET['id'] ?? 0;
            if (!$id) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'Income ID is required']);
                break;
            }
            
            $stmt = $conn->prepare("DELETE FROM income WHERE id=?");
            $result = $stmt->execute([$id]);
            $rowCount = $stmt->rowCount();
            
            if ($rowCount > 0) {
                echo json_encode(['success' => true, 'message' => 'Income deleted successfully', 'deleted' => $rowCount]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Income not found or already deleted']);
            }
            break;
    }
}

/**
 * Generate JWT token
 */
function generateToken($user) {
    $header = json_encode(['typ' => 'JWT', 'alg' => 'HS256']);
    $payload = json_encode([
        'user_id' => $user['id'],
        'email' => $user['email'],
        'role' => $user['role'],
        'iat' => time(),
        'exp' => time() + (60 * 60 * 24) // Token expires in 24 hours
    ]);
    
    $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
    $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
    $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, JWT_SECRET, true);
    $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
    
    return $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
}

/**
 * Verify JWT token
 */
function verifyToken($token) {
    try {
        $tokenParts = explode('.', $token);
        if (count($tokenParts) !== 3) return false;
        
        $header = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[0]));
        $payload = base64_decode(str_replace(['-', '_'], ['+', '/'], $tokenParts[1]));
        $signature = hash_hmac('sha256', $tokenParts[0] . "." . $tokenParts[1], JWT_SECRET, true);
        $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        if ($base64UrlSignature !== $tokenParts[2]) return false;
        
        $payload = json_decode($payload, true);
        if (isset($payload['exp']) && $payload['exp'] < time()) return false;
        
        return $payload;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Get authorization header
 */
function getAuthorizationHeader() {
    $headers = null;
    if (isset($_SERVER['Authorization'])) {
        $headers = trim($_SERVER["Authorization"]);
    } else if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $headers = trim($_SERVER["HTTP_AUTHORIZATION"]);
    } elseif (function_exists('apache_request_headers')) {
        $requestHeaders = apache_request_headers();
        $requestHeaders = array_combine(array_map('ucwords', array_keys($requestHeaders)), array_values($requestHeaders));
        if (isset($requestHeaders['Authorization'])) {
            $headers = trim($requestHeaders['Authorization']);
        }
    }
    return $headers;
}

/**
 * Get access token from header
 */
function getBearerToken() {
    $headers = getAuthorizationHeader();
    if (!empty($headers) && preg_match('/Bearer\s(\S+)/', $headers, $matches)) {
        return $matches[1];
    }
    return null;
}

/**
 * Check if user is authenticated
 */
function isAuthenticated() {
    $token = getBearerToken();
    if (!$token) return false;
    
    $decoded = verifyToken($token);
    if (!$decoded) return false;
    
    return $decoded;
}

/**
 * Require authentication
 */
function requireAuth() {
    $user = isAuthenticated();
    if (!$user) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized - Authentication required']);
        exit();
    }
    return $user;
}

/**
 * Handle authentication requests
 */
function handleAuth($conn, $method) {
    switch($method) {
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Input validation
            if (!isset($data['email']) || !isset($data['password'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Email and password are required']);
                return;
            }
            
            // Get user from database
            $stmt = $conn->prepare("SELECT id, username, email, full_name, role, password FROM users WHERE email = ?");
            $stmt->execute([$data['email']]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            // Verify password
            if ($user && password_verify($data['password'], $user['password'])) {
                // Remove sensitive data before creating token
                unset($user['password']);
                
                // Generate JWT token
                $token = generateToken($user);
                
                // Update last login (commented out - column doesn't exist yet)
                // $updateStmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                // $updateStmt->execute([$user['id']]);
                
                // Return user data with token
                echo json_encode([
                    'success' => true,
                    'token' => $token,
                    'user' => [
                        'id' => $user['id'],
                        'username' => $user['username'],
                        'email' => $user['email'],
                        'name' => $user['full_name'], // Add name field
                        'full_name' => $user['full_name'],
                        'role' => $user['role']
                    ]
                ]);
            } else {
                http_response_code(401);
                echo json_encode(['error' => 'Invalid email or password']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle user registration requests
 */
function handleRegister($conn, $method) {
    switch($method) {
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Input validation
            if (!isset($data['name']) || !isset($data['email']) || !isset($data['password'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Name, email and password are required']);
                return;
            }
            
            // Validate email format
            if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid email format']);
                return;
            }
            
            // Check if user already exists
            $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$data['email']]);
            if ($stmt->fetch()) {
                http_response_code(409);
                echo json_encode(['error' => 'User with this email already exists']);
                return;
            }
            
            // Hash password
            $hashedPassword = password_hash($data['password'], PASSWORD_DEFAULT);
            
            // Set default role if not provided
            $role = $data['role'] ?? 'teacher';
            
            // Validate role
            $validRoles = ['admin', 'teacher', 'finance', 'principal'];
            if (!in_array($role, $validRoles)) {
                $role = 'teacher';
            }
            
            // Create username from email
            $username = explode('@', $data['email'])[0];
            
            // Insert new user
            try {
                $stmt = $conn->prepare("
                    INSERT INTO users (username, email, password, full_name, role, phone, is_active, created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, 1, NOW())
                ");
                $stmt->execute([
                    $username,
                    $data['email'],
                    $hashedPassword,
                    $data['name'],
                    $role,
                    $data['phone'] ?? ''
                ]);
                
                $userId = $conn->lastInsertId();
                
                // Get the created user
                $stmt = $conn->prepare("SELECT id, username, email, full_name, role FROM users WHERE id = ?");
                $stmt->execute([$userId]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Generate JWT token
                $token = generateToken($user);
                
                // Return user data with token
                echo json_encode([
                    'success' => true,
                    'token' => $token,
                    'user' => [
                        'id' => $user['id'],
                        'username' => $user['username'],
                        'email' => $user['email'],
                        'full_name' => $user['full_name'],
                        'role' => $user['role']
                    ],
                    'message' => 'Registration successful'
                ]);
            } catch (PDOException $e) {
                http_response_code(500);
                echo json_encode(['error' => 'Failed to create user: ' . $e->getMessage()]);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle reports requests
 */
function handleReports($conn, $method) {
    switch($method) {
        case 'GET':
            // Return empty reports array for now
            // This can be expanded later with actual report generation
            echo json_encode([
                'success' => true,
                'data' => []
            ]);
            break;

        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            // Basic report generation placeholder
            $reportData = [
                'id' => uniqid(),
                'title' => $data['title'] ?? 'Untitled Report',
                'type' => $data['type'] ?? 'general',
                'generated_at' => date('Y-m-d H:i:s'),
                'data' => []
            ];
            
            echo json_encode([
                'success' => true,
                'data' => $reportData
            ]);
            break;

        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle academics requests - routes to academic handler
 */
function handleAcademics($conn, $method) {
    try {
        // Get sub-endpoint from query string
        $endpoint = $_GET['endpoint'] ?? $_GET['action'] ?? '';
        
        // Remove cache-busting parameters from endpoint
        $endpoint = preg_replace('/\?.*$/', '', $endpoint);
        
        error_log("Academic Request: Method=$method, Endpoint=$endpoint");
        
        // Get request data
        $data = null;
        if (in_array($method, ['POST', 'PUT'])) {
            $rawInput = file_get_contents('php://input');
            $data = json_decode($rawInput, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid JSON in request body: ' . json_last_error_msg());
            }
        } else {
            $data = $_GET;
        }
        
        // Route to academic handler
        if (function_exists('handleAcademicRequest')) {
            $result = handleAcademicRequest($method, $endpoint, $data);
            
            // Ensure we have a valid result
            if (!is_array($result)) {
                throw new Exception('Invalid response from academic handler');
            }
            
            echo json_encode($result);
        } else {
            http_response_code(501);
            echo json_encode(['success' => false, 'message' => 'Academic system not implemented']);
        }
    } catch (Exception $e) {
        error_log("Academic Handler Error: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false, 
            'message' => $e->getMessage(),
            'error' => 'Internal server error'
        ]);
    }
}

/**
 * Handle feeding records requests
 */
function handleFeeding($conn, $method) {
    switch($method) {
        case 'GET':
            $date = $_GET['date'] ?? date('Y-m-d');
            $class = $_GET['class'] ?? '';
            
            if (empty($class)) {
                http_response_code(400);
                echo json_encode(['error' => 'Class parameter is required']);
                return;
            }
            
            // Get students in the class with custom feeding fees, payment frequency, and scholarship status
            $stmt = $conn->prepare("SELECT student_id AS studentId, CONCAT(first_name, ' ', last_name) as name, class, custom_feeding_fee, feeding_fee_note, feeding_payment_frequency, feeding_weekly_balance, feeding_monthly_balance, feeding_last_monthly_payment, feeding_scholarship, feeding_scholarship_note FROM students WHERE class = ? AND status = 'active' ORDER BY first_name, last_name");
            $stmt->execute([$class]);
            $students = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get feeding records for the date and class
            $stmt = $conn->prepare("SELECT * FROM feeding_records WHERE feeding_date = ? AND class_name = ?");
            $stmt->execute([$date, $class]);
            $records = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Create a map of student records
            $recordsMap = [];
            foreach ($records as $record) {
                $recordsMap[$record['student_id']] = $record;
            }
            
            // Get class feeding fee
            $stmt = $conn->prepare("SELECT daily_fee FROM class_feeding_fees WHERE class_name = ?");
            $stmt->execute([$class]);
            $feeData = $stmt->fetch(PDO::FETCH_ASSOC);
            $dailyFee = $feeData ? $feeData['daily_fee'] : 0;
            
            // Combine student data with feeding records
            $result = [];
            foreach ($students as $student) {
                $studentId = $student['studentId'];
                $record = $recordsMap[$studentId] ?? null;
                
                // Use custom fee if set, otherwise use class default
                $studentFee = $student['custom_feeding_fee'] ?? $dailyFee;
                
                // Determine payment frequency (default to weekly if not set)
                $paymentFrequency = $student['feeding_payment_frequency'] ?? 'weekly';
                $weeklyBalance = (float)($student['feeding_weekly_balance'] ?? 0);
                $monthlyBalance = (float)($student['feeding_monthly_balance'] ?? 0);
                $lastMonthlyPayment = $student['feeding_last_monthly_payment'] ?? null;
                
                // Calculate display fee based on payment frequency
                $displayFee = $studentFee;
                $feeNote = $student['feeding_fee_note'] ?? null;
                
                if ($paymentFrequency === 'monthly') {
                    // For monthly payers, show the monthly amount (5 days * daily fee)
                    $monthlyAmount = $studentFee * 5;
                    $feeNote = $feeNote ?? "Monthly: GH₵" . number_format($monthlyAmount, 2);
                }
                
                // Check scholarship status
                $isScholarship = (bool)($student['feeding_scholarship'] ?? false);
                $scholarshipNote = $student['feeding_scholarship_note'] ?? null;
                
                // If on scholarship, fee is 0
                if ($isScholarship) {
                    $studentFee = 0;
                    $feeNote = $scholarshipNote ?? 'Scholarship';
                }
                
                $result[] = [
                    'student_id' => $studentId,
                    'student_name' => $student['name'],
                    'class_name' => $student['class'],
                    'is_present' => $record ? (bool)$record['is_present'] : false,
                    'has_paid' => $record ? (bool)$record['has_paid'] : false,
                    'amount_paid' => $record ? (float)$record['amount_paid'] : 0,
                    'notes' => $record ? $record['notes'] : null,
                    'daily_fee' => (float)$studentFee,
                    'custom_fee' => $student['custom_feeding_fee'] ? (float)$student['custom_feeding_fee'] : null,
                    'fee_note' => $feeNote,
                    'payment_frequency' => $paymentFrequency,
                    'weekly_balance' => $weeklyBalance,
                    'monthly_balance' => $monthlyBalance,
                    'last_monthly_payment' => $lastMonthlyPayment,
                    'is_scholarship' => $isScholarship,
                    'scholarship_note' => $scholarshipNote
                ];
            }
            
            echo json_encode([
                'success' => true,
                'data' => $result,
                'date' => $date,
                'class' => $class,
                'daily_fee' => (float)$dailyFee
            ]);
            break;
            
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['student_id']) || !isset($data['feeding_date']) || !isset($data['class_name'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing required fields']);
                return;
            }
            
            // Get student details including payment frequency
            $stmt = $conn->prepare("SELECT CONCAT(first_name, ' ', last_name) as name, custom_feeding_fee, feeding_payment_frequency, feeding_weekly_balance, feeding_monthly_balance, feeding_last_monthly_payment FROM students WHERE student_id = ?");
            $stmt->execute([$data['student_id']]);
            $student = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$student) {
                http_response_code(404);
                echo json_encode(['error' => 'Student not found']);
                return;
            }
            
            if (empty($student['name'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Student name is missing in database']);
                return;
            }
            
            // Get daily fee for the class
            $stmt = $conn->prepare("SELECT daily_fee FROM class_feeding_fees WHERE class_name = ?");
            $stmt->execute([$data['class_name']]);
            $feeData = $stmt->fetch(PDO::FETCH_ASSOC);
            $dailyFee = $feeData ? $feeData['daily_fee'] : 0;
            
            // Use custom fee if set
            $studentDailyFee = $student['custom_feeding_fee'] ?? $dailyFee;
            
            $isPresent = isset($data['is_present']) ? (bool)$data['is_present'] : false;
            $hasPaid = isset($data['has_paid']) ? (bool)$data['has_paid'] : false;
            
            // Handle payment based on frequency
            $paymentFrequency = $student['feeding_payment_frequency'] ?? 'daily';
            $weeklyBalance = (float)($student['feeding_weekly_balance'] ?? 0);
            $monthlyBalance = (float)($student['feeding_monthly_balance'] ?? 0);
            $customAmount = isset($data['custom_amount']) && $data['custom_amount'] > 0 ? (float)$data['custom_amount'] : null;
            
            $amountPaid = 0;
            $updateWeeklyBalance = false;
            $updateMonthlyBalance = false;
            $newWeeklyBalance = $weeklyBalance;
            $newMonthlyBalance = $monthlyBalance;
            
            if ($hasPaid && $isPresent) {
                if ($paymentFrequency === 'monthly') {
                    // Monthly payment logic
                    if ($customAmount) {
                        // New monthly payment received - distribute across 5 days (1 week)
                        $monthlyAmount = $customAmount;
                        $newMonthlyBalance = $monthlyAmount;
                        $amountPaid = $studentDailyFee; // Deduct daily amount
                        $newMonthlyBalance -= $amountPaid;
                        $updateMonthlyBalance = true;
                        
                        // Update last monthly payment date
                        $stmt = $conn->prepare("UPDATE students SET feeding_last_monthly_payment = ? WHERE student_id = ?");
                        $stmt->execute([$data['feeding_date'], $data['student_id']]);
                    } else {
                        // Deduct from existing balance
                        if ($monthlyBalance >= $studentDailyFee) {
                            $amountPaid = $studentDailyFee;
                            $newMonthlyBalance = $monthlyBalance - $amountPaid;
                            $updateMonthlyBalance = true;
                        } else {
                            // Insufficient balance
                            http_response_code(400);
                            echo json_encode([
                                'error' => 'Insufficient monthly balance. Please add monthly payment.',
                                'current_balance' => $monthlyBalance,
                                'required' => $studentDailyFee
                            ]);
                            return;
                        }
                    }
                } elseif ($paymentFrequency === 'weekly') {
                    // Weekly payment logic
                    if ($customAmount) {
                        // New weekly payment received - distribute across 5 days
                        $weeklyAmount = $customAmount;
                        $newWeeklyBalance = $weeklyAmount;
                        $amountPaid = $studentDailyFee; // Deduct daily amount
                        $newWeeklyBalance -= $amountPaid;
                        $updateWeeklyBalance = true;
                    } else {
                        // Deduct from existing balance
                        if ($weeklyBalance >= $studentDailyFee) {
                            $amountPaid = $studentDailyFee;
                            $newWeeklyBalance = $weeklyBalance - $amountPaid;
                            $updateWeeklyBalance = true;
                        } else {
                            // Insufficient balance
                            http_response_code(400);
                            echo json_encode([
                                'error' => 'Insufficient weekly balance. Please add weekly payment.',
                                'current_balance' => $weeklyBalance,
                                'required' => $studentDailyFee
                            ]);
                            return;
                        }
                    }
                } else {
                    // Daily payment
                    $amountPaid = $customAmount ?? $studentDailyFee;
                }
            }
            
            // Update weekly balance if needed
            if ($updateWeeklyBalance) {
                $stmt = $conn->prepare("UPDATE students SET feeding_weekly_balance = ? WHERE student_id = ?");
                $stmt->execute([$newWeeklyBalance, $data['student_id']]);
            }
            
            // Update monthly balance if needed
            if ($updateMonthlyBalance) {
                $stmt = $conn->prepare("UPDATE students SET feeding_monthly_balance = ? WHERE student_id = ?");
                $stmt->execute([$newMonthlyBalance, $data['student_id']]);
            }
            
            // Get notes if provided
            $notes = $data['notes'] ?? null;
            
            // Insert or update feeding record
            $stmt = $conn->prepare("
                INSERT INTO feeding_records (student_id, student_name, class_name, feeding_date, is_present, has_paid, amount_paid, notes, recorded_by)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                is_present = VALUES(is_present),
                has_paid = VALUES(has_paid),
                amount_paid = VALUES(amount_paid),
                notes = VALUES(notes),
                recorded_by = VALUES(recorded_by),
                updated_at = CURRENT_TIMESTAMP
            ");
            
            $stmt->execute([
                $data['student_id'],
                $student['name'],
                $data['class_name'],
                $data['feeding_date'],
                $isPresent,
                $hasPaid,
                $amountPaid,
                $notes,
                $data['recorded_by'] ?? 'System'
            ]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Feeding record updated successfully',
                'amount_paid' => $amountPaid,
                'new_balance' => $paymentFrequency === 'monthly' ? $newBalance : null,
                'payment_frequency' => $paymentFrequency
            ]);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle feeding fees management
 */
function handleFeedingFees($conn, $method) {
    switch($method) {
        case 'GET':
            $stmt = $conn->query("SELECT * FROM class_feeding_fees ORDER BY class_name");
            $fees = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'success' => true,
                'data' => $fees
            ]);
            break;
            
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['class_name']) || !isset($data['daily_fee'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Class name and daily fee are required']);
                return;
            }
            
            $stmt = $conn->prepare("
                INSERT INTO class_feeding_fees (class_name, daily_fee)
                VALUES (?, ?)
                ON DUPLICATE KEY UPDATE
                daily_fee = VALUES(daily_fee),
                updated_at = CURRENT_TIMESTAMP
            ");
            
            $stmt->execute([$data['class_name'], $data['daily_fee']]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Feeding fee updated successfully'
            ]);
            break;
            
        case 'PUT':
            // Get class name from query parameter (id)
            $className = $_GET['id'] ?? null;
            
            if (!$className) {
                http_response_code(400);
                echo json_encode(['error' => 'Class name (id) is required']);
                return;
            }
            
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['daily_fee'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Daily fee is required']);
                return;
            }
            
            // Update the feeding fee for the class
            $stmt = $conn->prepare("
                UPDATE class_feeding_fees 
                SET daily_fee = ?, updated_at = CURRENT_TIMESTAMP
                WHERE class_name = ?
            ");
            
            $stmt->execute([$data['daily_fee'], $className]);
            
            if ($stmt->rowCount() > 0) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Feeding fee updated successfully'
                ]);
            } else {
                // If no rows updated, try to insert
                $stmt = $conn->prepare("
                    INSERT INTO class_feeding_fees (class_name, daily_fee)
                    VALUES (?, ?)
                ");
                $stmt->execute([$className, $data['daily_fee']]);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Feeding fee created successfully'
                ]);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle feeding statistics requests
 */
function handleFeedingStats($conn, $method) {
    switch($method) {
        case 'GET':
            $startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
            $endDate = $_GET['end_date'] ?? date('Y-m-d');
            
            // Get overall statistics
            $stmt = $conn->prepare("
                SELECT 
                    COUNT(DISTINCT feeding_date) as total_days,
                    COUNT(*) as total_records,
                    SUM(CASE WHEN is_present = 1 THEN 1 ELSE 0 END) as total_present,
                    SUM(CASE WHEN has_paid = 1 THEN 1 ELSE 0 END) as total_paid_count,
                    SUM(amount_paid) as total_amount_collected
                FROM feeding_records
                WHERE feeding_date BETWEEN ? AND ?
            ");
            $stmt->execute([$startDate, $endDate]);
            $overall = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Get statistics by class
            $stmt = $conn->prepare("
                SELECT 
                    class_name,
                    COUNT(DISTINCT feeding_date) as days_recorded,
                    COUNT(*) as total_records,
                    SUM(CASE WHEN is_present = 1 THEN 1 ELSE 0 END) as students_present,
                    SUM(CASE WHEN has_paid = 1 THEN 1 ELSE 0 END) as students_paid,
                    SUM(amount_paid) as amount_collected,
                    AVG(amount_paid) as avg_amount
                FROM feeding_records
                WHERE feeding_date BETWEEN ? AND ?
                GROUP BY class_name
                ORDER BY class_name
            ");
            $stmt->execute([$startDate, $endDate]);
            $byClass = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get daily statistics for the period
            $stmt = $conn->prepare("
                SELECT 
                    feeding_date,
                    COUNT(*) as total_records,
                    SUM(CASE WHEN is_present = 1 THEN 1 ELSE 0 END) as students_present,
                    SUM(CASE WHEN has_paid = 1 THEN 1 ELSE 0 END) as students_paid,
                    SUM(amount_paid) as amount_collected
                FROM feeding_records
                WHERE feeding_date BETWEEN ? AND ?
                GROUP BY feeding_date
                ORDER BY feeding_date DESC
                LIMIT 30
            ");
            $stmt->execute([$startDate, $endDate]);
            $daily = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get students with custom fees
            $stmt = $conn->query("
                SELECT 
                    student_id,
                    CONCAT(first_name, ' ', last_name) as student_name,
                    class,
                    custom_feeding_fee,
                    feeding_fee_note
                FROM students
                WHERE custom_feeding_fee IS NOT NULL
                ORDER BY class, first_name
            ");
            $customFees = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get students on scholarship
            $stmt = $conn->query("
                SELECT 
                    student_id,
                    CONCAT(first_name, ' ', last_name) as student_name,
                    class,
                    feeding_scholarship_note
                FROM students
                WHERE feeding_scholarship = 1 AND status = 'active'
                ORDER BY class, first_name
            ");
            $scholarshipStudents = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get payment frequency breakdown
            $stmt = $conn->query("
                SELECT 
                    student_id,
                    CONCAT(first_name, ' ', last_name) as student_name,
                    class,
                    feeding_payment_frequency,
                    feeding_weekly_balance,
                    feeding_monthly_balance
                FROM students
                WHERE status = 'active'
                ORDER BY feeding_payment_frequency, class, first_name
            ");
            $paymentFrequency = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // Get recent records with notes (last 30 days)
            $stmt = $conn->prepare("
                SELECT 
                    fr.student_id,
                    fr.student_name,
                    fr.class_name,
                    fr.feeding_date,
                    fr.notes,
                    fr.has_paid
                FROM feeding_records fr
                WHERE fr.notes IS NOT NULL 
                AND fr.notes != ''
                AND fr.feeding_date BETWEEN ? AND ?
                ORDER BY fr.feeding_date DESC, fr.class_name, fr.student_name
                LIMIT 100
            ");
            $stmt->execute([$startDate, $endDate]);
            $studentsWithNotes = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'success' => true,
                'data' => [
                    'period' => [
                        'start_date' => $startDate,
                        'end_date' => $endDate
                    ],
                    'overall' => [
                        'total_days' => (int)$overall['total_days'],
                        'total_records' => (int)$overall['total_records'],
                        'total_present' => (int)$overall['total_present'],
                        'total_paid_count' => (int)$overall['total_paid_count'],
                        'total_amount_collected' => (float)$overall['total_amount_collected']
                    ],
                    'by_class' => array_map(function($row) {
                        return [
                            'class_name' => $row['class_name'],
                            'days_recorded' => (int)$row['days_recorded'],
                            'total_records' => (int)$row['total_records'],
                            'students_present' => (int)$row['students_present'],
                            'students_paid' => (int)$row['students_paid'],
                            'amount_collected' => (float)$row['amount_collected'],
                            'avg_amount' => (float)$row['avg_amount']
                        ];
                    }, $byClass),
                    'daily' => array_map(function($row) {
                        return [
                            'date' => $row['feeding_date'],
                            'total_records' => (int)$row['total_records'],
                            'students_present' => (int)$row['students_present'],
                            'students_paid' => (int)$row['students_paid'],
                            'amount_collected' => (float)$row['amount_collected']
                        ];
                    }, $daily),
                    'custom_fees' => array_map(function($row) {
                        return [
                            'student_id' => $row['student_id'],
                            'student_name' => $row['student_name'],
                            'class' => $row['class'],
                            'custom_fee' => (float)$row['custom_feeding_fee'],
                            'note' => $row['feeding_fee_note']
                        ];
                    }, $customFees),
                    'scholarship_students' => array_map(function($row) {
                        return [
                            'student_id' => $row['student_id'],
                            'student_name' => $row['student_name'],
                            'class' => $row['class'],
                            'note' => $row['feeding_scholarship_note']
                        ];
                    }, $scholarshipStudents),
                    'students_with_notes' => array_map(function($row) {
                        return [
                            'student_id' => $row['student_id'],
                            'student_name' => $row['student_name'],
                            'class' => $row['class_name'],
                            'date' => $row['feeding_date'],
                            'notes' => $row['notes'],
                            'has_paid' => (bool)$row['has_paid']
                        ];
                    }, $studentsWithNotes),
                    'payment_frequency' => array_map(function($row) {
                        return [
                            'student_id' => $row['student_id'],
                            'student_name' => $row['student_name'],
                            'class' => $row['class'],
                            'frequency' => $row['feeding_payment_frequency'] ?? 'daily',
                            'weekly_balance' => (float)($row['feeding_weekly_balance'] ?? 0),
                            'monthly_balance' => (float)($row['feeding_monthly_balance'] ?? 0)
                        ];
                    }, $paymentFrequency)
                ]
            ]);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle feeding class details requests
 */
function handleFeedingClassDetails($conn, $method) {
    switch($method) {
        case 'GET':
            $className = $_GET['class'] ?? '';
            $startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
            $endDate = $_GET['end_date'] ?? date('Y-m-d');
            
            if (empty($className)) {
                http_response_code(400);
                echo json_encode(['error' => 'Class parameter is required']);
                return;
            }
            
            // Get all feeding records for this class in the date range
            // Include ALL students (present or not) - show unpaid regardless of presence
            $stmt = $conn->prepare("
                SELECT 
                    fr.student_id,
                    fr.student_name,
                    fr.feeding_date,
                    fr.is_present,
                    COALESCE(fr.has_paid, 0) as has_paid,
                    COALESCE(fr.amount_paid, 0) as amount_paid,
                    fr.custom_fee,
                    cff.daily_fee as class_daily_fee
                FROM feeding_records fr
                LEFT JOIN class_feeding_fees cff ON fr.class_name = cff.class_name
                WHERE fr.class_name = ?
                AND fr.feeding_date BETWEEN ? AND ?
                ORDER BY fr.has_paid ASC, fr.feeding_date DESC, fr.student_name ASC
            ");
            $stmt->execute([$className, $startDate, $endDate]);
            $records = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode([
                'success' => true,
                'data' => array_map(function($record) {
                    // Ensure has_paid is properly converted to boolean
                    // Handle null, 0, "0", false, "" as false
                    $hasPaid = !empty($record['has_paid']) && $record['has_paid'] !== '0' && $record['has_paid'] !== 0;
                    
                    return [
                        'student_id' => $record['student_id'],
                        'student_name' => $record['student_name'],
                        'feeding_date' => $record['feeding_date'],
                        'is_present' => (bool)$record['is_present'],
                        'has_paid' => $hasPaid,
                        'amount_paid' => (float)($record['amount_paid'] ?? 0),
                        'daily_fee' => $record['custom_fee'] ?? $record['class_daily_fee'] ?? 0
                    ];
                }, $records)
            ]);
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle student feeding settings (payment frequency, custom fees)
 */
function handleFeedingStudentSettings($conn, $method) {
    switch($method) {
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($data['student_id'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Student ID is required']);
                return;
            }
            
            $studentId = $data['student_id'];
            $updates = [];
            $params = [];
            
            // Update payment frequency
            if (isset($data['payment_frequency'])) {
                if (!in_array($data['payment_frequency'], ['daily', 'weekly', 'monthly'])) {
                    http_response_code(400);
                    echo json_encode(['error' => 'Invalid payment frequency. Must be "daily", "weekly" or "monthly"']);
                    return;
                }
                $updates[] = "feeding_payment_frequency = ?";
                $params[] = $data['payment_frequency'];
            }
            
            // Update custom feeding fee
            if (isset($data['custom_fee'])) {
                $updates[] = "custom_feeding_fee = ?";
                $params[] = $data['custom_fee'] > 0 ? $data['custom_fee'] : null;
            }
            
            // Update fee note
            if (isset($data['fee_note'])) {
                $updates[] = "feeding_fee_note = ?";
                $params[] = $data['fee_note'];
            }
            
            // Update weekly balance (when adding weekly payment)
            if (isset($data['weekly_balance'])) {
                $updates[] = "feeding_weekly_balance = ?";
                $params[] = $data['weekly_balance'];
            }
            
            // Update monthly balance (when adding monthly payment)
            if (isset($data['monthly_balance'])) {
                $updates[] = "feeding_monthly_balance = ?";
                $params[] = $data['monthly_balance'];
            }
            
            // Update scholarship status
            if (isset($data['is_scholarship'])) {
                $updates[] = "feeding_scholarship = ?";
                $params[] = (bool)$data['is_scholarship'];
            }
            
            // Update scholarship note
            if (isset($data['scholarship_note'])) {
                $updates[] = "feeding_scholarship_note = ?";
                $params[] = $data['scholarship_note'];
            }
            
            if (empty($updates)) {
                http_response_code(400);
                echo json_encode(['error' => 'No valid fields to update']);
                return;
            }
            
            $params[] = $studentId;
            $sql = "UPDATE students SET " . implode(', ', $updates) . " WHERE student_id = ?";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute($params);
            
            if ($stmt->rowCount() > 0) {
                // Get updated student data
                $stmt = $conn->prepare("SELECT student_id, CONCAT(first_name, ' ', last_name) as name, custom_feeding_fee, feeding_fee_note, feeding_payment_frequency, feeding_weekly_balance, feeding_monthly_balance, feeding_scholarship, feeding_scholarship_note FROM students WHERE student_id = ?");
                $stmt->execute([$studentId]);
                $student = $stmt->fetch(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'success' => true,
                    'message' => 'Student feeding settings updated successfully',
                    'data' => [
                        'student_id' => $student['student_id'],
                        'student_name' => $student['name'],
                        'custom_fee' => $student['custom_feeding_fee'] ? (float)$student['custom_feeding_fee'] : null,
                        'fee_note' => $student['feeding_fee_note'],
                        'payment_frequency' => $student['feeding_payment_frequency'] ?? 'daily',
                        'weekly_balance' => (float)($student['feeding_weekly_balance'] ?? 0),
                        'monthly_balance' => (float)($student['feeding_monthly_balance'] ?? 0),
                        'is_scholarship' => (bool)($student['feeding_scholarship'] ?? false),
                        'scholarship_note' => $student['feeding_scholarship_note']
                    ]
                ]);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Student not found or no changes made']);
            }
            break;
            
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
}

/**
 * Handle clearing all feeding records (Admin only)
 */
function handleClearAllFeedingRecords($conn, $method) {
    if ($method !== 'DELETE') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        return;
    }

    // Verify authentication and admin role
    $headers = getallheaders();
    $authHeader = $headers['Authorization'] ?? '';
    
    if (empty($authHeader)) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized - No token provided']);
        return;
    }

    $token = str_replace('Bearer ', '', $authHeader);
    $payload = verifyJWT($token);
    
    if (!$payload) {
        http_response_code(401);
        echo json_encode(['error' => 'Unauthorized - Invalid token']);
        return;
    }

    // Check if user is admin or super_admin
    if (!isset($payload['role']) || !in_array($payload['role'], ['admin', 'super_admin'])) {
        http_response_code(403);
        echo json_encode(['error' => 'Forbidden - Admin access required']);
        return;
    }

    try {
        // Count records before deletion
        $stmt = $conn->query("SELECT COUNT(*) as count FROM feeding_records");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $deletedCount = $result['count'];

        // Delete all feeding records
        $conn->exec("DELETE FROM feeding_records");

        echo json_encode([
            'success' => true,
            'message' => 'All feeding records cleared successfully',
            'deleted_count' => $deletedCount
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'error' => 'Failed to clear feeding records',
            'details' => $e->getMessage()
        ]);
    }
}

/**
 * Ensure feeding-related tables exist
 */
function ensureFeedingTables($conn) {
    try {
    // Create class_feeding_fees table if it doesn't exist
    $conn->exec("
        CREATE TABLE IF NOT EXISTS class_feeding_fees (
            id INT AUTO_INCREMENT PRIMARY KEY,
            class_name VARCHAR(100) NOT NULL UNIQUE,
            daily_fee DECIMAL(10,2) NOT NULL DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    // Create feeding_records table if it doesn't exist
    $conn->exec("
        CREATE TABLE IF NOT EXISTS feeding_records (
            id INT AUTO_INCREMENT PRIMARY KEY,
            student_id VARCHAR(50) NOT NULL,
            student_name VARCHAR(255) NOT NULL,
            class_name VARCHAR(100) NOT NULL,
            feeding_date DATE NOT NULL,
            is_present TINYINT(1) NOT NULL DEFAULT 0,
            has_paid TINYINT(1) NOT NULL DEFAULT 0,
            amount_paid DECIMAL(10,2) NOT NULL DEFAULT 0,
            recorded_by VARCHAR(100) DEFAULT 'System',
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_student_date (student_id, feeding_date),
            INDEX idx_date_class (feeding_date, class_name),
            INDEX idx_student_date (student_id, feeding_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");

    // Seed some default class fees if table is empty
    $conn->exec("
        INSERT IGNORE INTO class_feeding_fees (class_name, daily_fee) VALUES
            ('Creche', 5.00),
            ('Nursery 1', 5.50),
            ('Nursery 2', 5.50),
            ('KG 1', 6.00),
            ('KG 2', 6.00),
            ('Class 1', 6.50),
            ('Class 2', 6.50),
            ('Class 3', 7.00),
            ('Class 4', 7.00),
            ('Class 5', 7.50),
            ('Class 6', 7.50),
            ('Basic 1', 6.50),
            ('Basic 2', 6.50),
            ('Basic 3', 7.00),
            ('Basic 4', 7.00),
            ('Basic 5', 7.50),
            ('Basic 6', 7.50),
            ('JHS 1', 8.00),
            ('JHS 2', 8.00),
            ('JHS 3', 8.50);
    ");
    } catch (PDOException $e) {
        // Log error but don't crash
        error_log("Error creating feeding tables: " . $e->getMessage());
        throw $e; // Re-throw so caller can handle
    }
}
?>