<?php
/**
 * Complete Database Fix - All Issues
 * Run this: https://4102009823.ceiscy.com/school/database/complete_fix.php
 */

require_once __DIR__ . '/../config/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Complete Database Fix</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 1000px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #4CAF50; padding: 8px 12px; background: #e8f5e9; margin: 3px 0; border-radius: 4px; border-left: 4px solid #4CAF50; }
        .error { color: #f44336; padding: 8px 12px; background: #ffebee; margin: 3px 0; border-radius: 4px; border-left: 4px solid #f44336; }
        .info { color: #2196F3; padding: 8px 12px; background: #e3f2fd; margin: 3px 0; border-radius: 4px; border-left: 4px solid #2196F3; }
        h1 { color: #333; margin-top: 0; }
        h2 { color: #666; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; margin-top: 25px; margin-bottom: 15px; }
        .btn { display: inline-block; padding: 12px 24px; background: #4CAF50; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; font-weight: bold; }
        .step { background: #f9f9f9; padding: 15px; margin: 10px 0; border-radius: 4px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🚀 Complete Database Fix</h1>
        <p>This will fix all database issues for the feeding system.</p>
        
<?php

try {
    $pdo = getDBConnection();
    echo "<div class='success'>✅ Database connected successfully</div>";
    
    $totalFixed = 0;
    $totalErrors = 0;
    
    // STEP 1: Get current feeding_records structure
    echo "<div class='step'>";
    echo "<h2>Step 1: Analyze feeding_records Table</h2>";
    
    $currentCols = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_COLUMN);
    echo "<div class='info'>Current columns: " . implode(', ', $currentCols) . "</div>";
    
    // STEP 2: Add missing columns one by one
    echo "<h2>Step 2: Add Missing Columns</h2>";
    
    $columnsToAdd = [
        ['name' => 'date', 'def' => 'DATE NOT NULL', 'after' => 'student_id'],
        ['name' => 'status', 'def' => "ENUM('present', 'absent') DEFAULT 'present'", 'after' => 'date'],
        ['name' => 'amount_paid', 'def' => 'DECIMAL(10,2) DEFAULT 0.00', 'after' => 'status'],
        ['name' => 'payment_method', 'def' => "ENUM('cash', 'bank_transfer', 'mobile_money', 'cheque') NULL", 'after' => 'amount_paid'],
        ['name' => 'notes', 'def' => 'TEXT NULL', 'after' => 'payment_method'],
        ['name' => 'created_at', 'def' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP', 'after' => 'notes'],
        ['name' => 'updated_at', 'def' => 'TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP', 'after' => 'created_at']
    ];
    
    foreach ($columnsToAdd as $col) {
        if (!in_array($col['name'], $currentCols)) {
            try {
                // Check if the AFTER column exists
                if ($col['after'] && !in_array($col['after'], $currentCols)) {
                    // Add without AFTER clause
                    $sql = "ALTER TABLE feeding_records ADD COLUMN {$col['name']} {$col['def']}";
                } else {
                    $sql = "ALTER TABLE feeding_records ADD COLUMN {$col['name']} {$col['def']} AFTER {$col['after']}";
                }
                
                $pdo->exec($sql);
                echo "<div class='success'>✅ Added: {$col['name']}</div>";
                $totalFixed++;
                $currentCols[] = $col['name']; // Update current columns list
            } catch (Exception $e) {
                echo "<div class='error'>❌ {$col['name']}: " . $e->getMessage() . "</div>";
                $totalErrors++;
            }
        } else {
            echo "<div class='info'>✓ {$col['name']} exists</div>";
        }
    }
    echo "</div>";
    
    // STEP 3: Verify students table
    echo "<div class='step'>";
    echo "<h2>Step 3: Verify Students Table</h2>";
    
    $studentCols = $pdo->query("SHOW COLUMNS FROM students")->fetchAll(PDO::FETCH_COLUMN);
    $requiredStudentCols = [
        'feeding_balance',
        'feeding_weekly_balance',
        'feeding_payment_frequency',
        'custom_feeding_fee',
        'feeding_scholarship_percentage',
        'feeding_notes'
    ];
    
    $allPresent = true;
    foreach ($requiredStudentCols as $col) {
        if (in_array($col, $studentCols)) {
            echo "<div class='info'>✓ $col</div>";
        } else {
            echo "<div class='error'>❌ Missing: $col</div>";
            $allPresent = false;
        }
    }
    
    if ($allPresent) {
        echo "<div class='success'>✅ All students columns present</div>";
    }
    echo "</div>";
    
    // STEP 4: Final verification
    echo "<div class='step'>";
    echo "<h2>Step 4: Final Verification</h2>";
    
    $finalCols = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_COLUMN);
    $requiredFeedingCols = ['id', 'student_id', 'date', 'status', 'amount_paid', 'payment_method', 'notes', 'created_at', 'updated_at'];
    
    $missingCols = array_diff($requiredFeedingCols, $finalCols);
    
    if (empty($missingCols)) {
        echo "<div class='success' style='font-size: 16px; padding: 15px;'>
            ✅ <strong>feeding_records table is complete!</strong><br>
            All " . count($requiredFeedingCols) . " required columns are present.
        </div>";
    } else {
        echo "<div class='error'>
            ❌ Still missing: " . implode(', ', $missingCols) . "
        </div>";
    }
    
    echo "<div class='info'>
        <strong>Final columns:</strong><br>
        " . implode(', ', $finalCols) . "
    </div>";
    echo "</div>";
    
    // Summary
    echo "<h2>📊 Summary</h2>";
    echo "<div class='info'>
        • Columns added: $totalFixed<br>
        • Errors: $totalErrors<br>
        • Status: " . (empty($missingCols) && $totalErrors === 0 ? "✅ READY" : "⚠️ NEEDS ATTENTION") . "
    </div>";
    
    if (empty($missingCols) && $totalErrors === 0) {
        echo "<div class='success' style='font-size: 18px; padding: 20px; margin-top: 20px;'>
            🎉 <strong>All Done!</strong><br><br>
            Your database is now properly configured.<br>
            The feeding system should work correctly now.
        </div>";
        
        echo "<a href='../' class='btn'>← Back to Application</a>";
    } else {
        echo "<div class='error' style='padding: 20px; margin-top: 20px;'>
            ⚠️ Some issues remain. Please contact support or check the error messages above.
        </div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>
        <strong>❌ Fatal Error:</strong><br>
        " . htmlspecialchars($e->getMessage()) . "
    </div>";
}

?>
    </div>
</body>
</html>
