<?php
/**
 * Create All Feeding System Tables
 * Run: https://4102009823.ceiscy.com/school/database/create_feeding_tables.php
 */

require_once __DIR__ . '/../config/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Create Feeding Tables</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 900px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #4CAF50; padding: 10px; background: #e8f5e9; margin: 5px 0; border-radius: 4px; }
        .error { color: #f44336; padding: 10px; background: #ffebee; margin: 5px 0; border-radius: 4px; }
        .info { color: #2196F3; padding: 10px; background: #e3f2fd; margin: 5px 0; border-radius: 4px; }
        h1 { color: #333; }
        h2 { color: #666; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; margin-top: 25px; }
        .btn { display: inline-block; padding: 12px 24px; background: #4CAF50; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Create Feeding System Tables</h1>
        
<?php

try {
    $pdo = getDBConnection();
    echo "<div class='success'>✅ Connected to database</div>";
    
    // 1. Create class_feeding_fees table
    echo "<h2>1. Class Feeding Fees Table</h2>";
    
    $createClassFees = "
    CREATE TABLE IF NOT EXISTS class_feeding_fees (
        id INT AUTO_INCREMENT PRIMARY KEY,
        class_name VARCHAR(50) NOT NULL UNIQUE,
        daily_fee DECIMAL(10,2) NOT NULL DEFAULT 0.00,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    $pdo->exec($createClassFees);
    echo "<div class='success'>✅ class_feeding_fees table created</div>";
    
    // Insert default fees for common classes
    $defaultClasses = [
        'Class 1' => 5.00,
        'Class 2' => 5.00,
        'Class 3' => 5.00,
        'Class 4' => 5.00,
        'Class 5' => 5.00,
        'Class 6' => 5.00,
        'JHS 1' => 6.00,
        'JHS 2' => 6.00,
        'JHS 3' => 6.00
    ];
    
    $insertStmt = $pdo->prepare("
        INSERT INTO class_feeding_fees (class_name, daily_fee)
        VALUES (?, ?)
        ON DUPLICATE KEY UPDATE daily_fee = daily_fee
    ");
    
    foreach ($defaultClasses as $className => $fee) {
        $insertStmt->execute([$className, $fee]);
    }
    echo "<div class='info'>✓ Added default fees for " . count($defaultClasses) . " classes</div>";
    
    // 2. Create/recreate feeding_records table
    echo "<h2>2. Feeding Records Table</h2>";
    
    // Check if table exists
    $tableExists = $pdo->query("SHOW TABLES LIKE 'feeding_records'")->fetch();
    
    if ($tableExists) {
        echo "<div class='info'>Table exists - checking structure...</div>";
        
        // Check if it has the correct columns
        $columns = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_COLUMN);
        $requiredCols = ['student_id', 'student_name', 'class_name', 'feeding_date', 'is_present', 'has_paid', 'amount_paid', 'notes'];
        $missingCols = array_diff($requiredCols, $columns);
        
        if (!empty($missingCols)) {
            echo "<div class='error'>❌ Table has wrong structure. Recreating...</div>";
            
            // Backup data
            $backup = $pdo->query("SELECT * FROM feeding_records")->fetchAll(PDO::FETCH_ASSOC);
            echo "<div class='info'>📦 Backed up " . count($backup) . " records</div>";
            
            // Drop and recreate
            $pdo->exec("DROP TABLE feeding_records");
            
            $createFeedingRecords = "
            CREATE TABLE feeding_records (
                id INT AUTO_INCREMENT PRIMARY KEY,
                student_id VARCHAR(50) NOT NULL,
                student_name VARCHAR(255) NOT NULL,
                class_name VARCHAR(50) NOT NULL,
                feeding_date DATE NOT NULL,
                is_present BOOLEAN DEFAULT FALSE,
                has_paid BOOLEAN DEFAULT FALSE,
                amount_paid DECIMAL(10,2) DEFAULT 0.00,
                notes TEXT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_student_date (student_id, feeding_date),
                INDEX idx_date_class (feeding_date, class_name)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $pdo->exec($createFeedingRecords);
            echo "<div class='success'>✅ Recreated feeding_records table</div>";
            
            // Restore data
            if (!empty($backup)) {
                $restored = 0;
                $insertStmt = $pdo->prepare("
                    INSERT INTO feeding_records 
                    (student_id, student_name, class_name, feeding_date, is_present, has_paid, amount_paid, notes)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                foreach ($backup as $record) {
                    try {
                        // Get student info
                        $stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', last_name) as name, class FROM students WHERE student_id = ?");
                        $stmt->execute([$record['student_id']]);
                        $student = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($student) {
                            $insertStmt->execute([
                                $record['student_id'],
                                $student['name'],
                                $student['class'],
                                $record['date'] ?? $record['feeding_date'],
                                $record['is_present'] ?? ($record['status'] === 'present' ? 1 : 0),
                                $record['has_paid'] ?? ($record['amount_paid'] > 0 ? 1 : 0),
                                $record['amount_paid'] ?? 0,
                                $record['notes'] ?? null
                            ]);
                            $restored++;
                        }
                    } catch (Exception $e) {
                        // Skip problematic records
                    }
                }
                echo "<div class='success'>✅ Restored $restored records</div>";
            }
        } else {
            echo "<div class='success'>✅ feeding_records table structure is correct</div>";
        }
    } else {
        // Create new table
        $createFeedingRecords = "
        CREATE TABLE feeding_records (
            id INT AUTO_INCREMENT PRIMARY KEY,
            student_id VARCHAR(50) NOT NULL,
            student_name VARCHAR(255) NOT NULL,
            class_name VARCHAR(50) NOT NULL,
            feeding_date DATE NOT NULL,
            is_present BOOLEAN DEFAULT FALSE,
            has_paid BOOLEAN DEFAULT FALSE,
            amount_paid DECIMAL(10,2) DEFAULT 0.00,
            notes TEXT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_student_date (student_id, feeding_date),
            INDEX idx_date_class (feeding_date, class_name)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $pdo->exec($createFeedingRecords);
        echo "<div class='success'>✅ Created feeding_records table</div>";
    }
    
    // 3. Verify all tables
    echo "<h2>3. Verification</h2>";
    
    $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_COLUMN);
    
    if (in_array('class_feeding_fees', $tables)) {
        echo "<div class='success'>✅ class_feeding_fees exists</div>";
        
        $count = $pdo->query("SELECT COUNT(*) FROM class_feeding_fees")->fetchColumn();
        echo "<div class='info'>• Contains $count class fees</div>";
    }
    
    if (in_array('feeding_records', $tables)) {
        echo "<div class='success'>✅ feeding_records exists</div>";
        
        $count = $pdo->query("SELECT COUNT(*) FROM feeding_records")->fetchColumn();
        echo "<div class='info'>• Contains $count records</div>";
    }
    
    echo "<div class='success' style='font-size: 18px; padding: 20px; margin-top: 20px;'>
        🎉 <strong>All Done!</strong><br><br>
        Your feeding system database is now properly set up.<br>
        Go back to your application and test the feeding features.
    </div>";
    
    echo "<a href='../' class='btn'>← Back to Application</a>";
    
} catch (Exception $e) {
    echo "<div class='error'><strong>❌ Error:</strong><br>" . htmlspecialchars($e->getMessage()) . "</div>";
}

?>
    </div>
</body>
</html>
