<?php
/**
 * Fix Assessment Constraints to 50/50
 * Changes score limits from 40/60 to 50/50
 */

require_once __DIR__ . '/../config/database.php';

echo "=== Fixing Assessment Score Constraints (50/50) ===\n\n";

try {
    // Step 1: Check for existing scores that would violate new constraints
    echo "1. Checking existing scores...\n";
    $stmt = $conn->query("
        SELECT COUNT(*) as count
        FROM student_assessments
        WHERE class_score > 50 OR exam_score > 50
    ");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $violationCount = $result['count'];
    
    if ($violationCount > 0) {
        echo "   Found {$violationCount} records with scores > 50\n";
        echo "   Adjusting scores to fit new limits...\n";
        
        // Cap scores at 50
        $conn->exec("UPDATE student_assessments SET class_score = 50.00 WHERE class_score > 50");
        $conn->exec("UPDATE student_assessments SET exam_score = 50.00 WHERE exam_score > 50");
        echo "   ✅ Adjusted scores to maximum of 50\n\n";
    } else {
        echo "   ✅ No scores exceed 50\n\n";
    }
    
    // Step 2: Drop old constraints
    echo "2. Dropping old constraints (40/60)...\n";
    try {
        $conn->exec("ALTER TABLE student_assessments DROP CONSTRAINT CONSTRAINT_1");
        echo "   ✅ Dropped CONSTRAINT_1 (class_score <= 40)\n";
    } catch (Exception $e) {
        echo "   ⚠️  CONSTRAINT_1 not found (may already be dropped)\n";
    }
    
    try {
        $conn->exec("ALTER TABLE student_assessments DROP CONSTRAINT CONSTRAINT_2");
        echo "   ✅ Dropped CONSTRAINT_2 (exam_score <= 60)\n\n";
    } catch (Exception $e) {
        echo "   ⚠️  CONSTRAINT_2 not found (may already be dropped)\n\n";
    }
    
    // Step 3: Add new constraints
    echo "3. Adding new constraints (50/50)...\n";
    try {
        $conn->exec("ALTER TABLE student_assessments ADD CONSTRAINT check_class_score CHECK (class_score <= 50.00)");
        echo "   ✅ Added check_class_score (class_score <= 50)\n";
    } catch (Exception $e) {
        echo "   ⚠️  check_class_score may already exist\n";
    }
    
    try {
        $conn->exec("ALTER TABLE student_assessments ADD CONSTRAINT check_exam_score CHECK (exam_score <= 50.00)");
        echo "   ✅ Added check_exam_score (exam_score <= 50)\n\n";
    } catch (Exception $e) {
        echo "   ⚠️  check_exam_score may already exist\n\n";
    }
    
    // Step 4: Verify
    echo "4. Verifying constraints...\n";
    $stmt = $conn->query("
        SELECT 
            CONSTRAINT_NAME,
            CHECK_CLAUSE
        FROM information_schema.CHECK_CONSTRAINTS
        WHERE CONSTRAINT_SCHEMA = DATABASE()
        AND TABLE_NAME = 'student_assessments'
        AND CONSTRAINT_NAME IN ('check_class_score', 'check_exam_score')
    ");
    
    $constraints = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (count($constraints) === 2) {
        foreach ($constraints as $row) {
            echo "   ✅ {$row['CONSTRAINT_NAME']}: {$row['CHECK_CLAUSE']}\n";
        }
    } else {
        echo "   ⚠️  Expected 2 constraints, found " . count($constraints) . "\n";
    }
    
    echo "\n=== Success! ===\n";
    echo "Assessment scores are now limited to:\n";
    echo "- Class score: 0-50 ✅\n";
    echo "- Exam score: 0-50 ✅\n";
    echo "- Total: 0-100 ✅\n";
    
} catch (Exception $e) {
    echo "\n❌ Error: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
}
?>
