<?php
/**
 * Fix feeding_records Table to Match API Expectations
 * Run: https://4102009823.ceiscy.com/school/database/fix_feeding_records_structure.php
 */

require_once __DIR__ . '/../config/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Feeding Records Structure</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 900px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #4CAF50; padding: 10px; background: #e8f5e9; margin: 5px 0; border-radius: 4px; }
        .error { color: #f44336; padding: 10px; background: #ffebee; margin: 5px 0; border-radius: 4px; }
        .info { color: #2196F3; padding: 10px; background: #e3f2fd; margin: 5px 0; border-radius: 4px; }
        h1 { color: #333; }
        h2 { color: #666; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; margin-top: 25px; }
        .btn { display: inline-block; padding: 12px 24px; background: #4CAF50; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Fix Feeding Records Table Structure</h1>
        
<?php

try {
    $pdo = getDBConnection();
    echo "<div class='success'>✅ Connected to database</div>";
    
    // Drop and recreate the table with correct structure
    echo "<h2>Recreating feeding_records Table</h2>";
    
    try {
        // Backup existing data if any
        $backupData = [];
        try {
            $stmt = $pdo->query("SELECT * FROM feeding_records");
            $backupData = $stmt->fetchAll(PDO::FETCH_ASSOC);
            echo "<div class='info'>📦 Backed up " . count($backupData) . " existing records</div>";
        } catch (Exception $e) {
            echo "<div class='info'>No existing data to backup</div>";
        }
        
        // Drop the table
        $pdo->exec("DROP TABLE IF EXISTS feeding_records");
        echo "<div class='success'>✅ Dropped old table</div>";
        
        // Create new table with correct structure
        $createTable = "
        CREATE TABLE feeding_records (
            id INT AUTO_INCREMENT PRIMARY KEY,
            student_id VARCHAR(50) NOT NULL,
            student_name VARCHAR(255) NOT NULL,
            class_name VARCHAR(50) NOT NULL,
            feeding_date DATE NOT NULL,
            is_present BOOLEAN DEFAULT FALSE,
            has_paid BOOLEAN DEFAULT FALSE,
            amount_paid DECIMAL(10,2) DEFAULT 0.00,
            notes TEXT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY unique_student_date (student_id, feeding_date),
            INDEX idx_date_class (feeding_date, class_name),
            INDEX idx_student (student_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $pdo->exec($createTable);
        echo "<div class='success'>✅ Created new table with correct structure</div>";
        
        // Restore data if any (with column mapping)
        if (!empty($backupData)) {
            $restored = 0;
            foreach ($backupData as $record) {
                try {
                    // Map old columns to new columns
                    $studentId = $record['student_id'] ?? null;
                    $date = $record['date'] ?? $record['feeding_date'] ?? null;
                    $status = $record['status'] ?? 'present';
                    $isPresent = ($status === 'present') ? 1 : 0;
                    $amountPaid = $record['amount_paid'] ?? 0;
                    $hasPaid = ($amountPaid > 0) ? 1 : 0;
                    $notes = $record['notes'] ?? null;
                    
                    if ($studentId && $date) {
                        // Get student name and class
                        $stmt = $pdo->prepare("SELECT CONCAT(first_name, ' ', last_name) as name, class FROM students WHERE student_id = ?");
                        $stmt->execute([$studentId]);
                        $student = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        if ($student) {
                            $insertStmt = $pdo->prepare("
                                INSERT INTO feeding_records 
                                (student_id, student_name, class_name, feeding_date, is_present, has_paid, amount_paid, notes)
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                            ");
                            $insertStmt->execute([
                                $studentId,
                                $student['name'],
                                $student['class'],
                                $date,
                                $isPresent,
                                $hasPaid,
                                $amountPaid,
                                $notes
                            ]);
                            $restored++;
                        }
                    }
                } catch (Exception $e) {
                    // Skip records that can't be restored
                }
            }
            echo "<div class='success'>✅ Restored $restored records</div>";
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Error: " . $e->getMessage() . "</div>";
    }
    
    // Verify final structure
    echo "<h2>Final Structure</h2>";
    $columns = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<div class='info'><strong>Columns:</strong><br>";
    foreach ($columns as $col) {
        echo "• {$col['Field']} ({$col['Type']})<br>";
    }
    echo "</div>";
    
    // Check if all required columns exist
    $columnNames = array_column($columns, 'Field');
    $required = ['id', 'student_id', 'student_name', 'class_name', 'feeding_date', 'is_present', 'has_paid', 'amount_paid', 'notes'];
    $missing = array_diff($required, $columnNames);
    
    if (empty($missing)) {
        echo "<div class='success' style='font-size: 18px; padding: 20px; margin-top: 20px;'>
            🎉 <strong>Perfect!</strong><br><br>
            The feeding_records table now has the correct structure.<br>
            Your feeding system should work properly now.
        </div>";
        
        echo "<a href='../' class='btn'>← Back to Application</a>";
    } else {
        echo "<div class='error'>Still missing: " . implode(', ', $missing) . "</div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'><strong>❌ Fatal Error:</strong><br>" . htmlspecialchars($e->getMessage()) . "</div>";
}

?>
    </div>
</body>
</html>
