<?php
/**
 * Fix Feeding System Database Issues
 * Run this: https://4102009823.ceiscy.com/school/database/fix_feeding_system.php
 */

require_once __DIR__ . '/../config/database.php';

header('Content-Type: text/html; charset=utf-8');
?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Feeding System - Divine Grace School</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 900px; margin: 20px auto; padding: 20px; background: #f5f5f5; }
        .container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        .success { color: #4CAF50; padding: 10px; background: #e8f5e9; margin: 5px 0; border-radius: 4px; border-left: 4px solid #4CAF50; }
        .error { color: #f44336; padding: 10px; background: #ffebee; margin: 5px 0; border-radius: 4px; border-left: 4px solid #f44336; }
        .info { color: #2196F3; padding: 10px; background: #e3f2fd; margin: 5px 0; border-radius: 4px; border-left: 4px solid #2196F3; }
        h1 { color: #333; margin-top: 0; }
        h2 { color: #666; border-bottom: 2px solid #4CAF50; padding-bottom: 10px; margin-top: 30px; }
        .btn { display: inline-block; padding: 12px 24px; background: #4CAF50; color: white; text-decoration: none; border-radius: 4px; margin-top: 20px; }
        .btn:hover { background: #45a049; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Fix Feeding System Database</h1>
        
<?php

try {
    $pdo = getDBConnection();
    echo "<div class='success'>✅ Database connected successfully</div>";
    
    $fixed = [];
    $errors = [];
    
    // 1. Fix students table columns
    echo "<h2>Step 1: Students Table Columns</h2>";
    
    $studentColumns = [
        'feeding_balance' => "DECIMAL(10,2) DEFAULT 0.00",
        'feeding_weekly_balance' => "DECIMAL(10,2) DEFAULT 0.00",
        'feeding_payment_frequency' => "ENUM('daily', 'weekly', 'monthly', 'termly') DEFAULT 'daily'",
        'custom_feeding_fee' => "DECIMAL(10,2) NULL",
        'feeding_scholarship_percentage' => "INT DEFAULT 0",
        'feeding_notes' => "TEXT NULL"
    ];
    
    foreach ($studentColumns as $column => $definition) {
        try {
            $check = $pdo->query("SHOW COLUMNS FROM students LIKE '$column'")->fetch();
            if (!$check) {
                $pdo->exec("ALTER TABLE students ADD COLUMN $column $definition");
                echo "<div class='success'>✅ Added: $column</div>";
                $fixed[] = "students.$column";
            } else {
                echo "<div class='info'>✓ Already exists: $column</div>";
            }
        } catch (Exception $e) {
            echo "<div class='error'>❌ Error adding $column: " . $e->getMessage() . "</div>";
            $errors[] = "students.$column: " . $e->getMessage();
        }
    }
    
    // 2. Create or fix feeding_records table
    echo "<h2>Step 2: Feeding Records Table</h2>";
    
    try {
        // Check if table exists
        $tableExists = $pdo->query("SHOW TABLES LIKE 'feeding_records'")->fetch();
        
        if (!$tableExists) {
            // Create table
            $createTable = "
            CREATE TABLE feeding_records (
                id INT AUTO_INCREMENT PRIMARY KEY,
                student_id INT NOT NULL,
                date DATE NOT NULL,
                status ENUM('present', 'absent') DEFAULT 'present',
                amount_paid DECIMAL(10,2) DEFAULT 0.00,
                payment_method ENUM('cash', 'bank_transfer', 'mobile_money', 'cheque') NULL,
                notes TEXT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (student_id) REFERENCES students(id) ON DELETE CASCADE,
                UNIQUE KEY unique_student_date (student_id, date)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $pdo->exec($createTable);
            echo "<div class='success'>✅ Created feeding_records table</div>";
            $fixed[] = "feeding_records table created";
        } else {
            echo "<div class='info'>✓ feeding_records table exists</div>";
            
            // Check and add notes column if missing
            $notesExists = $pdo->query("SHOW COLUMNS FROM feeding_records LIKE 'notes'")->fetch();
            if (!$notesExists) {
                $pdo->exec("ALTER TABLE feeding_records ADD COLUMN notes TEXT NULL AFTER payment_method");
                echo "<div class='success'>✅ Added notes column to feeding_records</div>";
                $fixed[] = "feeding_records.notes";
            } else {
                echo "<div class='info'>✓ notes column exists</div>";
            }
            
            // Check and add payment_method column if missing
            $paymentMethodExists = $pdo->query("SHOW COLUMNS FROM feeding_records LIKE 'payment_method'")->fetch();
            if (!$paymentMethodExists) {
                $pdo->exec("ALTER TABLE feeding_records ADD COLUMN payment_method ENUM('cash', 'bank_transfer', 'mobile_money', 'cheque') NULL AFTER amount_paid");
                echo "<div class='success'>✅ Added payment_method column to feeding_records</div>";
                $fixed[] = "feeding_records.payment_method";
            } else {
                echo "<div class='info'>✓ payment_method column exists</div>";
            }
        }
    } catch (Exception $e) {
        echo "<div class='error'>❌ Error with feeding_records table: " . $e->getMessage() . "</div>";
        $errors[] = "feeding_records: " . $e->getMessage();
    }
    
    // 3. Verify structure
    echo "<h2>Step 3: Verification</h2>";
    
    try {
        // Check students table
        $studentCols = $pdo->query("SHOW COLUMNS FROM students")->fetchAll(PDO::FETCH_COLUMN);
        $requiredStudentCols = ['feeding_balance', 'feeding_weekly_balance', 'feeding_payment_frequency', 'custom_feeding_fee', 'feeding_scholarship_percentage', 'feeding_notes'];
        $missingStudentCols = array_diff($requiredStudentCols, $studentCols);
        
        if (empty($missingStudentCols)) {
            echo "<div class='success'>✅ All students table columns present</div>";
        } else {
            echo "<div class='error'>❌ Missing students columns: " . implode(', ', $missingStudentCols) . "</div>";
        }
        
        // Check feeding_records table
        $feedingCols = $pdo->query("SHOW COLUMNS FROM feeding_records")->fetchAll(PDO::FETCH_COLUMN);
        $requiredFeedingCols = ['id', 'student_id', 'date', 'status', 'amount_paid', 'payment_method', 'notes'];
        $missingFeedingCols = array_diff($requiredFeedingCols, $feedingCols);
        
        if (empty($missingFeedingCols)) {
            echo "<div class='success'>✅ All feeding_records table columns present</div>";
        } else {
            echo "<div class='error'>❌ Missing feeding_records columns: " . implode(', ', $missingFeedingCols) . "</div>";
        }
        
    } catch (Exception $e) {
        echo "<div class='error'>❌ Verification error: " . $e->getMessage() . "</div>";
    }
    
    // 4. Summary
    echo "<h2>Summary</h2>";
    
    if (count($fixed) > 0) {
        echo "<div class='success'><strong>✅ Fixed " . count($fixed) . " items:</strong><br>";
        foreach ($fixed as $item) {
            echo "• $item<br>";
        }
        echo "</div>";
    }
    
    if (count($errors) > 0) {
        echo "<div class='error'><strong>❌ " . count($errors) . " errors:</strong><br>";
        foreach ($errors as $error) {
            echo "• $error<br>";
        }
        echo "</div>";
    }
    
    if (count($errors) === 0) {
        echo "<div class='success' style='font-size: 18px; padding: 20px;'>
            🎉 <strong>All Done!</strong><br><br>
            Your feeding system database is now properly configured.<br>
            Go back to your application and refresh the page.
        </div>";
        
        echo "<a href='../' class='btn'>← Back to Application</a>";
    } else {
        echo "<div class='error'>
            Some errors occurred. Please check the error messages above and try again.
        </div>";
    }
    
} catch (Exception $e) {
    echo "<div class='error'>
        <strong>❌ Fatal Error:</strong><br>
        " . htmlspecialchars($e->getMessage()) . "
    </div>";
    echo "<pre>" . htmlspecialchars($e->getTraceAsString()) . "</pre>";
}

?>
    </div>
</body>
</html>
