<?php
/**
 * Migration: Link Students to Academic Classes
 * Adds class_id foreign key to students table
 */

require_once __DIR__ . '/../config/database.php';

echo "🔗 Linking Students to Academic Classes...\n\n";

try {
    $conn = getDBConnection();
    
    // Step 1: Add class_id column to students table
    echo "1. Adding class_id column to students table...\n";
    $sql = "ALTER TABLE students ADD COLUMN IF NOT EXISTS class_id INT NULL AFTER class";
    $conn->query($sql);
    echo "   ✅ Column added\n\n";
    
    // Step 2: Populate class_id based on existing class names
    echo "2. Mapping existing students to academic classes...\n";
    
    // Get all classes from academic system
    $classesResult = $conn->query("SELECT id, class_name FROM classes WHERE is_active = 1");
    $classes = [];
    while ($row = $classesResult->fetch_assoc()) {
        $classes[$row['class_name']] = $row['id'];
    }
    
    // Update students with matching class names
    $updateCount = 0;
    foreach ($classes as $className => $classId) {
        $stmt = $conn->prepare("UPDATE students SET class_id = ? WHERE class = ? AND class_id IS NULL");
        $stmt->bind_param("is", $classId, $className);
        $stmt->execute();
        $affected = $stmt->affected_rows;
        if ($affected > 0) {
            echo "   ✅ Linked $affected students to $className (ID: $classId)\n";
            $updateCount += $affected;
        }
    }
    
    echo "\n   Total students linked: $updateCount\n\n";
    
    // Step 3: Add foreign key constraint
    echo "3. Adding foreign key constraint...\n";
    
    // First, check if constraint already exists
    $checkConstraint = $conn->query("
        SELECT CONSTRAINT_NAME 
        FROM information_schema.TABLE_CONSTRAINTS 
        WHERE TABLE_SCHEMA = DATABASE() 
        AND TABLE_NAME = 'students' 
        AND CONSTRAINT_NAME = 'fk_students_class'
    ");
    
    if ($checkConstraint->num_rows === 0) {
        $sql = "ALTER TABLE students 
                ADD CONSTRAINT fk_students_class 
                FOREIGN KEY (class_id) REFERENCES classes(id) 
                ON DELETE SET NULL 
                ON UPDATE CASCADE";
        $conn->query($sql);
        echo "   ✅ Foreign key constraint added\n\n";
    } else {
        echo "   ℹ️  Foreign key constraint already exists\n\n";
    }
    
    // Step 4: Add index for better performance
    echo "4. Adding index on class_id...\n";
    $checkIndex = $conn->query("
        SHOW INDEX FROM students WHERE Key_name = 'idx_class_id'
    ");
    
    if ($checkIndex->num_rows === 0) {
        $conn->query("ALTER TABLE students ADD INDEX idx_class_id (class_id)");
        echo "   ✅ Index added\n\n";
    } else {
        echo "   ℹ️  Index already exists\n\n";
    }
    
    // Step 5: Verify the migration
    echo "5. Verifying migration...\n";
    $result = $conn->query("
        SELECT 
            COUNT(*) as total_students,
            COUNT(class_id) as linked_students,
            COUNT(*) - COUNT(class_id) as unlinked_students
        FROM students
    ");
    $stats = $result->fetch_assoc();
    
    echo "   Total students: {$stats['total_students']}\n";
    echo "   Linked to classes: {$stats['linked_students']}\n";
    echo "   Not linked: {$stats['unlinked_students']}\n\n";
    
    if ($stats['unlinked_students'] > 0) {
        echo "   ⚠️  Some students are not linked to academic classes.\n";
        echo "   This is normal if they have class names that don't match academic classes.\n\n";
    }
    
    $conn->close();
    
    echo str_repeat("=", 50) . "\n";
    echo "✅ Migration Complete!\n";
    echo str_repeat("=", 50) . "\n\n";
    
    echo "Benefits:\n";
    echo "• Students are now linked to academic classes\n";
    echo "• Easier to fetch students by class\n";
    echo "• Better data integrity with foreign keys\n";
    echo "• Improved query performance with indexes\n\n";
    
    echo "Next steps:\n";
    echo "1. Update API to use class_id for queries\n";
    echo "2. Update frontend to use linked data\n";
    echo "3. Test student-class relationships\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    exit(1);
}
