<?php
/**
 * Academic System Database Setup Script
 * Run this to create all academic tables
 */

require_once __DIR__ . '/../config/database.php';

echo "🎓 Setting up Academic System Database...\n\n";

try {
    // Read SQL file
    $sql = file_get_contents(__DIR__ . '/create_academic_tables.sql');
    
    if ($sql === false) {
        throw new Exception("Failed to read SQL file");
    }
    
    // Split into individual statements
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($stmt) {
            return !empty($stmt) && !preg_match('/^\s*--/', $stmt);
        }
    );
    
    $conn = getDBConnection();
    $conn->begin_transaction();
    
    $successCount = 0;
    $errorCount = 0;
    
    foreach ($statements as $statement) {
        if (empty(trim($statement))) continue;
        
        try {
            $conn->query($statement);
            $successCount++;
            
            // Extract table name for logging
            if (preg_match('/CREATE TABLE.*?`?(\w+)`?/i', $statement, $matches)) {
                echo "✅ Created table: {$matches[1]}\n";
            } elseif (preg_match('/INSERT INTO\s+(\w+)/i', $statement, $matches)) {
                echo "✅ Inserted data into: {$matches[1]}\n";
            }
        } catch (Exception $e) {
            $errorCount++;
            echo "⚠️  Warning: " . $e->getMessage() . "\n";
        }
    }
    
    $conn->commit();
    
    echo "\n" . str_repeat("=", 50) . "\n";
    echo "✅ Academic System Setup Complete!\n";
    echo "📊 Statements executed: $successCount\n";
    if ($errorCount > 0) {
        echo "⚠️  Warnings: $errorCount\n";
    }
    echo str_repeat("=", 50) . "\n\n";
    
    // Verify tables
    echo "🔍 Verifying tables...\n";
    $tables = [
        'academic_terms',
        'classes',
        'subjects',
        'class_subjects',
        'assessment_types',
        'assessments',
        'student_scores',
        'grading_scale',
        'attendance',
        'subject_attendance',
        'report_cards',
        'report_card_details'
    ];
    
    foreach ($tables as $table) {
        $result = $conn->query("SHOW TABLES LIKE '$table'");
        if ($result->num_rows > 0) {
            echo "✅ $table\n";
        } else {
            echo "❌ $table (missing)\n";
        }
    }
    
    $conn->close();
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    if (isset($conn)) {
        $conn->rollback();
        $conn->close();
    }
    exit(1);
}

echo "\n🎉 Academic System is ready to use!\n";
